package gql_test

import (
	"errors"
	"fmt"
	"net/http"
	"strings"
	"testing"
	"time"

	"github.com/stretchr/testify/assert"

	"code.justin.tv/extensions/eastwatch/internal/api/gql"
	"code.justin.tv/extensions/eastwatch/internal/config"
	"code.justin.tv/extensions/eastwatch/internal/fultonlibs/FultonGoLangSimpleBootstrap/simplebootstrap"
	"code.justin.tv/extensions/eastwatch/internal/metrics"
	"code.justin.tv/extensions/eastwatch/internal/models/token"
	"code.justin.tv/extensions/eastwatch/internal/testutil"
	"code.justin.tv/extensions/eastwatch/internal/users"
	"github.com/stretchr/testify/require"
)

const (
	activeState         = "ACTIVE"
	inactiveState       = "INACTIVE"
	gqlServerErrorRetry = 10 // GQL may return server error occasionally on some endpoints
	lowVolumeRetry      = 5
	highVolumnRetry     = 20
)

var (
	// Reference: https://git-aws.internal.justin.tv/gds/gds/blob/master/extensions/ems/dynamicmanagement/hardcoded_pairings.go
	// TODO: Unable to test any cases with 2 Content-Matched extensions, since in staging there're currently only 1 record that is Content-Matched extension for beta
	//lint:ignore U1000 EXDSC-1110 ignore unused staticcheck failure for now
	contentMatchedExtensionIDs = []string{
		"xm9qmvnmls6ifd21kjejl0b30gyheo",
		"lgpf9j7y8n1ja9onkb6w7bxfkhk2zl",
	}
	nonContentMatchedExtensionID = "4xthrtbw4oqjxl478qsjspia8vt3ra"
	nonMatchedGame               = gql.Game{
		ID:   "5432",
		Name: "TechnoClash",
	}
	managedOn  = true
	managedOff = false
)

func TestDynamicManagementSuccess(t *testing.T) {
	t.Skip("[EXDSC-1110] test is brittle, causing Eastwatch to fail often")

	bs, client, userID, oauth := initiate(t)
	expectedInstallationID, expectedContentMatchedGames := getExpectedExtension(t, client, oauth, contentMatchedExtensionIDs[0], userID, true)

	// Make sure everything gets clean up
	defer updateChannelGame(t, client, oauth, userID, "", "")
	defer uninstallExtension(t, client, oauth, expectedInstallationID)

	t.Run("test case 1: switching between content-matched and non content-matched games", func(t *testing.T) {
		/******** Test Case 1: Switching between content-matched and non content-matched games
			Initial channel setting: no game
			1. Install and activate a Content-Matched extension -> Dynamic Management on, extension activated
			2. Switch channel game to the Content-Matched game -> Dynamic Management on, extension activated
			3. Switch channel game to non Content-Matched game -> Dynamic Management on, extension deactivated
			4. Switch channel game back to the Content-Matched game -> Dynamic Management on, extension activated
			5. Manually deactivate the Content-Matched extension -> Dynamic Management off, extension deactivated
		********/
		defer updateChannelGame(t, client, oauth, userID, "", "")
		defer uninstallExtension(t, client, oauth, expectedInstallationID)

		// Initial channel setting: no game
		updateChannelGame(t, client, oauth, userID, "", "")

		// 1. Install and activate a Content-Matched extension -> Dynamic Management on, extension activated
		installExtension(t, client, oauth, expectedInstallationID)
		validateState(t, client, oauth, expectedContentMatchedGames, expectedInstallationID, inactiveState, nil, lowVolumeRetry)
		applyExtensionActivation(t, client, oauth, userID, expectedInstallationID, true)
		validateState(t, client, oauth, expectedContentMatchedGames, expectedInstallationID, activeState, &managedOn, lowVolumeRetry)

		// 2. Switch channel game to the Content-Matched game -> Dynamic Management on, extension activated
		updateChannelGame(t, client, oauth, userID, expectedContentMatchedGames[0].Name, expectedContentMatchedGames[0].ID)
		validateState(t, client, oauth, expectedContentMatchedGames, expectedInstallationID, activeState, &managedOn, lowVolumeRetry)

		// 3. Switch channel game to non Content-Matched game -> Dynamic Management on, extension deactivated
		// Noticed that lambda takes time to process so do retries here
		updateChannelGame(t, client, oauth, userID, nonMatchedGame.Name, nonMatchedGame.ID)
		validateState(t, client, oauth, expectedContentMatchedGames, expectedInstallationID, inactiveState, &managedOn, highVolumnRetry)

		// 4. Switch channel game back to the Content-Matched game -> Dynamic Management on, extension activated
		// Noticed that lambda takes time to process so do retries here
		updateChannelGame(t, client, oauth, userID, expectedContentMatchedGames[0].Name, expectedContentMatchedGames[0].ID)
		validateState(t, client, oauth, expectedContentMatchedGames, expectedInstallationID, activeState, &managedOn, highVolumnRetry)

		// 5. Manually deactivate the Content-Matched extension -> Dynamic Management off, extension deactivated
		applyExtensionActivation(t, client, oauth, userID, expectedInstallationID, false)
		validateState(t, client, oauth, expectedContentMatchedGames, expectedInstallationID, inactiveState, &managedOff, lowVolumeRetry)
	})

	t.Run("test case 2: manually activate and deactivate content-matched extension", func(t *testing.T) {
		/******** Test Case 2: manually activate and deactivate content-matched extension
			Initial channel setting: Content-Matched game
			1. Install and activate a Content-Matched extension -> Dynamic Management on, extension activated
			2. Switch channel game to non Content-Matched game -> Dynamic Management on, extension deactivated
			3. Manually activate the Content-Matched extension -> Dynamic Management on, extension activated
			4. Manually deactivate the Content-Matched extension -> Dynamic Management off, extension deactivated
			5. Switch channel game to Content-Matched game -> Dynamic Management off, extension deactivated
		********/
		defer updateChannelGame(t, client, oauth, userID, "", "")
		defer uninstallExtension(t, client, oauth, expectedInstallationID)

		// Initial channel setting: Content-Matched game
		updateChannelGame(t, client, oauth, userID, expectedContentMatchedGames[0].Name, expectedContentMatchedGames[0].ID)
		time.Sleep(100 * time.Millisecond)

		// 1. Install and activate a Content-Matched extension -> Dynamic Management on, extension activated
		installExtension(t, client, oauth, expectedInstallationID)
		validateState(t, client, oauth, expectedContentMatchedGames, expectedInstallationID, inactiveState, nil, lowVolumeRetry)
		applyExtensionActivation(t, client, oauth, userID, expectedInstallationID, true)
		validateState(t, client, oauth, expectedContentMatchedGames, expectedInstallationID, activeState, &managedOn, lowVolumeRetry)

		// 2. Switch channel game to non Content-Matched game -> Dynamic Management on, extension deactivated
		// Noticed that lambda takes time to process so do retries here
		updateChannelGame(t, client, oauth, userID, nonMatchedGame.Name, nonMatchedGame.ID)
		validateState(t, client, oauth, expectedContentMatchedGames, expectedInstallationID, inactiveState, &managedOn, highVolumnRetry)

		// 3. Manually activate the Content-Matched extension -> Dynamic Management on, extension activated
		applyExtensionActivation(t, client, oauth, userID, expectedInstallationID, true)
		validateState(t, client, oauth, expectedContentMatchedGames, expectedInstallationID, activeState, &managedOn, lowVolumeRetry)

		// 4. Manually deactivate the Content-Matched extension -> Dynamic Management off, extension deactivated
		applyExtensionActivation(t, client, oauth, userID, expectedInstallationID, false)
		validateState(t, client, oauth, expectedContentMatchedGames, expectedInstallationID, inactiveState, &managedOff, lowVolumeRetry)

		// 5. Switch channel game to Content-Matched game -> Dynamic Management off, extension deactivated
		updateChannelGame(t, client, oauth, userID, expectedContentMatchedGames[0].Name, expectedContentMatchedGames[0].ID)
		validateState(t, client, oauth, expectedContentMatchedGames, expectedInstallationID, inactiveState, &managedOff, lowVolumeRetry)
	})

	t.Run("test case 3: Switch to content-matched game after manually deactivation", func(t *testing.T) {
		/******** Test Case 3: Switch to content-matched game after manually deactivation
			Initial channel setting: no game
			1. Install and activate a Content-Matched extension  -> Dynamic Management on, extension activated
			2. Manually deactivate the Content-Matched extension -> Dynamic Management off, extension deactivated
			3. Switch channel game to Content-Matched game -> Dynamic Management off, extension deactivated
		********/
		defer updateChannelGame(t, client, oauth, userID, "", "")
		defer uninstallExtension(t, client, oauth, expectedInstallationID)

		// Initial channel setting: no game
		updateChannelGame(t, client, oauth, userID, "", "")
		time.Sleep(100 * time.Millisecond)

		// 1. Install and activate a Content-Matched extension  -> Dynamic Management on, extension activated
		installExtension(t, client, oauth, expectedInstallationID)
		validateState(t, client, oauth, expectedContentMatchedGames, expectedInstallationID, inactiveState, nil, lowVolumeRetry)
		applyExtensionActivation(t, client, oauth, userID, expectedInstallationID, true)
		validateState(t, client, oauth, expectedContentMatchedGames, expectedInstallationID, activeState, &managedOn, lowVolumeRetry)

		// 2. Manually deactivate the Content-Matched extension -> Dynamic Management off, extension deactivated
		applyExtensionActivation(t, client, oauth, userID, expectedInstallationID, false)
		validateState(t, client, oauth, expectedContentMatchedGames, expectedInstallationID, inactiveState, &managedOff, lowVolumeRetry)

		// 3. Switch channel game to Content-Matched game -> Dynamic Management off, extension deactivated
		updateChannelGame(t, client, oauth, userID, expectedContentMatchedGames[0].Name, expectedContentMatchedGames[0].ID)
		validateState(t, client, oauth, expectedContentMatchedGames, expectedInstallationID, inactiveState, &managedOff, lowVolumeRetry)
	})

	t.Run("test case 4: Manually set feature flag on and off", func(t *testing.T) {
		/******** Test Case 4: Manually set feature flag on and off
			Initial channel setting: no game
			1. Install and activate a Content-Matched extension -> Dynamic Management on, extension activated
			2. Switch channel game to Content-Matched game -> Dynamic Management on, extension activated
			3. Manually set feature flag off -> Dynamic Management off, extension activated
			4. Switch channel game to non Content-Matched game -> Dynamic Management off, extension activated
			5. Manually deactivate the Content-Matched extension -> Dynamic Management off, extension deactivated
			6. Manually set feature flag on -> Dynamic Management on, extension deactivated
			7. Switch channel game to Content-Matched game -> Dynamic Management on, extension activated
		********/
		defer updateChannelGame(t, client, oauth, userID, "", "")
		defer uninstallExtension(t, client, oauth, expectedInstallationID)

		// Initial channel setting: no game
		updateChannelGame(t, client, oauth, userID, "", "")
		time.Sleep(100 * time.Millisecond)

		// 1. Install and activate a Content-Matched extension  -> Dynamic Management on, extension activated
		installExtension(t, client, oauth, expectedInstallationID)
		validateState(t, client, oauth, expectedContentMatchedGames, expectedInstallationID, inactiveState, nil, lowVolumeRetry)
		applyExtensionActivation(t, client, oauth, userID, expectedInstallationID, true)
		time.Sleep(1 * time.Second)
		validateState(t, client, oauth, expectedContentMatchedGames, expectedInstallationID, activeState, &managedOn, lowVolumeRetry)

		// 2. Switch channel game to Content-Matched game -> Dynamic Management on, extension activated
		updateChannelGame(t, client, oauth, userID, expectedContentMatchedGames[0].Name, expectedContentMatchedGames[0].ID)
		validateState(t, client, oauth, expectedContentMatchedGames, expectedInstallationID, activeState, &managedOn, lowVolumeRetry)

		// 3. Manually set feature flag off -> Dynamic Management off, extension activated
		setDynamicManagementFlag(t, client, oauth, expectedInstallationID, expectedContentMatchedGames[0].ID, managedOff)
		validateState(t, client, oauth, expectedContentMatchedGames, expectedInstallationID, activeState, &managedOff, lowVolumeRetry)

		// 4. Switch channel game to non Content-Matched game -> Dynamic Management off, extension activated
		updateChannelGame(t, client, oauth, userID, nonMatchedGame.Name, nonMatchedGame.ID)
		validateState(t, client, oauth, expectedContentMatchedGames, expectedInstallationID, activeState, &managedOff, lowVolumeRetry)

		// 5. Manually deactivate the Content-Matched extension -> Dynamic Management off, extension deactivated
		applyExtensionActivation(t, client, oauth, userID, expectedInstallationID, false)
		time.Sleep(1 * time.Second)
		validateState(t, client, oauth, expectedContentMatchedGames, expectedInstallationID, inactiveState, &managedOff, lowVolumeRetry)

		// 6. Manually set feature flag on -> Dynamic Management on, extension deactivated
		setDynamicManagementFlag(t, client, oauth, expectedInstallationID, expectedContentMatchedGames[0].ID, managedOn)
		validateState(t, client, oauth, expectedContentMatchedGames, expectedInstallationID, inactiveState, &managedOn, lowVolumeRetry)

		// 7. Switch channel game to Content-Matched game -> Dynamic Management on, extension activated
		// Noticed that lambda takes time to process so do retries here
		updateChannelGame(t, client, oauth, userID, expectedContentMatchedGames[0].Name, expectedContentMatchedGames[0].ID)
		validateState(t, client, oauth, expectedContentMatchedGames, expectedInstallationID, activeState, &managedOn, highVolumnRetry)
	})

	t.Run("test case 5: Activate contont-matched and non content-match games in the same slot", func(t *testing.T) {
		/******** Test Case 5: Activate contont-matched and non content-match games in the same slot
			Initial channel setting: no game
			1. Install and activate a Content-Matched extension (ext1) -> ext1 Dynamic Management on, extension activated
			2. Switch channel game to non Content-Matched game -> ext1 Dynamic Management on, extension deactivated
			3. Install and activate a non Content-Matched extension (ext2) -> ext2 no Dynamic Management, extension activated
			4. Switch channel game to Content-Matched game ->
				ext1 Dynamic Management on, extension activated;
				ext2 no Dynamic Management, extension deactivated
		********/
		defer updateChannelGame(t, client, oauth, userID, "", "")
		defer uninstallExtension(t, client, oauth, expectedInstallationID)

		// Initial channel setting: no game
		updateChannelGame(t, client, oauth, userID, "", "")
		time.Sleep(100 * time.Millisecond)

		// 1. Install and activate a Content-Matched extension (ext1) -> ext1 Dynamic Management on, extension activated
		installExtension(t, client, oauth, expectedInstallationID)
		validateState(t, client, oauth, expectedContentMatchedGames, expectedInstallationID, inactiveState, nil, lowVolumeRetry)
		applyExtensionActivation(t, client, oauth, userID, expectedInstallationID, true)
		validateState(t, client, oauth, expectedContentMatchedGames, expectedInstallationID, activeState, &managedOn, lowVolumeRetry)

		// 2. Switch channel game to non Content-Matched game -> ext1 Dynamic Management on, extension deactivated
		// Noticed that lambda takes time to process so do retries here
		updateChannelGame(t, client, oauth, userID, nonMatchedGame.Name, nonMatchedGame.ID)
		validateState(t, client, oauth, expectedContentMatchedGames, expectedInstallationID, inactiveState, &managedOn, highVolumnRetry)

		// 3. Install and activate a non Content-Matched extension (ext2) -> ext2 no Dynamic Management, extension activated
		expectedInstallationID2, _ := getExpectedExtension(t, client, oauth, nonContentMatchedExtensionID, userID, false)
		installExtension(t, client, oauth, expectedInstallationID2)
		defer uninstallExtension(t, client, oauth, expectedInstallationID2)
		validateState(t, client, oauth, nil, expectedInstallationID2, inactiveState, nil, lowVolumeRetry)
		applyExtensionActivation(t, client, oauth, userID, expectedInstallationID2, true)
		validateState(t, client, oauth, nil, expectedInstallationID2, activeState, nil, lowVolumeRetry)

		// 4. Switch channel game to Content-Matched game ->
		//		ext1 Dynamic Management on, extension activated;
		//		ext2 no Dynamic Management, extension deactivated
		// Noticed that lambda takes time to process so do retries here
		updateChannelGame(t, client, oauth, userID, expectedContentMatchedGames[0].Name, expectedContentMatchedGames[0].ID)
		validateState(t, client, oauth, expectedContentMatchedGames, expectedInstallationID, activeState, &managedOn, highVolumnRetry)
		validateState(t, client, oauth, nil, expectedInstallationID2, inactiveState, nil, highVolumnRetry)
	})

	bs.SampleReporter.Stop()
}

// 	=========================================
//	Utilities Functions
//	=========================================

//lint:ignore U1000 EXDSC-1110 ignore unused staticcheck failure for now
func initiate(t *testing.T) (*simplebootstrap.FultonBootstrap, gql.Client, string, token.OAuth) {
	// setup environment
	cfg := config.Staging
	bs, err := metrics.InitBootStrap("Eastwatch")
	if err != nil {
		panic(fmt.Sprintf("unable to initialize stats! %v", err))
	}
	captured := testutil.NewRoundTripRecorder(
		testutil.NewMetricsRoundTripper(http.DefaultTransport, bs))
	httpClient := &http.Client{
		Timeout:   2 * time.Second,
		Transport: captured,
	}

	client, err := gql.NewClient(httpClient, cfg.GQLURL)
	require.NoError(t, err)

	userID := users.User1.ID()
	oauth, err := users.User1.OAuthToken(cfg, config.SiteTwilight)
	require.NoError(t, err)

	return bs, client, userID, oauth
}

func getExpectedExtension(t *testing.T, client gql.Client, oauth token.OAuth, extensionID, userID string, isContentMatched bool) (string, []gql.Game) {
	var expectedExt gql.ExtensionDocument
	var expectedID, expectedVersion, expectedInstallationID string
	var expectedContentMatchedGames []gql.Game

	// Verify expecting extension
	expectedExt, err := client.GetExtensionByID(oauth, extensionID)
	require.NoError(t, err)
	expectedID, expectedVersion, err = parseManifestID(expectedExt.ID)
	require.NoError(t, err)
	assert.Equal(t, extensionID, expectedID)
	require.True(t, expectedExt.Views.VideoOverlay != nil || expectedExt.Views.Panel != nil || expectedExt.Views.Component != nil)
	expectedInstallationID = constructInstallationID(expectedID, expectedVersion, userID)
	if isContentMatched {
		require.NotNil(t, expectedExt.ContentMatchedGames)
		require.True(t, len(expectedExt.ContentMatchedGames) > 0)
		expectedContentMatchedGames = expectedExt.ContentMatchedGames
	}

	return expectedInstallationID, expectedContentMatchedGames
}

func installExtension(t *testing.T, client gql.Client, oauth token.OAuth, installationID string) {
	id, version, channelID, err := parseInstallationID(installationID)
	require.Nil(t, err)
	manifestID := constructManifestID(id, version)
	installedExtension, err := client.InstallExtension(oauth, manifestID, channelID)
	require.NoError(t, err)
	assert.Equal(t, installationID, installedExtension.InstallationID)
}

func uninstallExtension(t *testing.T, client gql.Client, oauth token.OAuth, installationID string) {
	_, err := client.UninstallExtension(oauth, installationID)
	require.NoError(t, err)
}

func applyExtensionActivation(t *testing.T, client gql.Client, oauth token.OAuth, userID, expectedInstallationID string, isActivation bool) gql.InstalledExtension {
	extensionActivationInput := gql.ExtensionActivationsInput{
		InstallationID: expectedInstallationID,
	}
	if isActivation {
		extensionActivationInput.VideoOverlay = &gql.VideoOverlayActivationInput{
			Slot: gql.VideoOverlaySlotExtension1,
		}
	}
	currentInstalledExtensions, err := client.ApplyExtensionActivations(oauth, userID, extensionActivationInput)
	require.NoError(t, err)
	assert.True(t, len(currentInstalledExtensions) > 0)
	var currentInstalledExtension gql.InstalledExtension
	for _, installedExt := range currentInstalledExtensions {
		if installedExt.InstallationID == expectedInstallationID {
			if isActivation {
				require.Equal(t, activeState, installedExt.ActivationConfig.State)
			} else {
				require.Equal(t, inactiveState, installedExt.ActivationConfig.State)
			}
			currentInstalledExtension = installedExt
			break
		}
	}
	require.NotEmpty(t, currentInstalledExtension)

	return currentInstalledExtension
}

func validateState(t *testing.T, client gql.Client, oauth token.OAuth, expectedContentMatchedGames []gql.Game, expectedInstallationID, expectedState string, expectedIsManaged *bool, retryAttempts int) {

	var currentInstalledExtension gql.InstalledExtension
	currentUser, err := client.GetCurrentUser(oauth)
	require.NoError(t, err)

	for _, ext := range currentUser.InstalledExtensions {
		if ext.InstallationID == expectedInstallationID {
			currentInstalledExtension = ext
			break
		}
	}
	require.NotEmpty(t, currentInstalledExtension)

	// use an eventually so we can retry every second
	assert.Eventually(t,
		func() bool {
			currentUser, err = client.GetCurrentUser(oauth)
			if err != nil {
				t.Logf("error getting current user from graphql: %v", err)
				return false
			}
			for _, ext := range currentUser.InstalledExtensions {
				if ext.InstallationID == expectedInstallationID {
					currentInstalledExtension = ext
					return currentInstalledExtension.ActivationConfig.State == expectedState
				}
			}
			return false
		},
		time.Duration(retryAttempts)*time.Second,
		1*time.Second,
		"retrieved installed extension activation state is %s, expected to be %s... this could be due to the dynamic management lambda not taking action yet",
		currentInstalledExtension.ActivationConfig.State,
		expectedState,
	)

	if expectedIsManaged != nil {
		require.NotEmpty(t, currentInstalledExtension.PermittedFeatures)
		require.NotNil(t, currentInstalledExtension.PermittedFeatures.DynamicManagement)
		for i, dm := range currentInstalledExtension.PermittedFeatures.DynamicManagement {
			assert.Equal(t, expectedContentMatchedGames[i].ID, dm.Game.ID)
			assert.Equal(t, *expectedIsManaged, dm.IsManaged)
		}
	} else {
		require.Nil(t, currentInstalledExtension.PermittedFeatures.DynamicManagement)
	}
}

func updateChannelGame(t *testing.T, client gql.Client, oauth token.OAuth, channelID, gameName, expectedGameID string) string {
	gameID, err := client.UpdateBroadcastSettings(oauth, channelID, gameName)
	if err != nil {
		for i := 0; i < gqlServerErrorRetry && err != nil; i++ {
			time.Sleep(100 * time.Millisecond)
			fmt.Printf("\tRetrying on gql.UpdateBroadcastSettings mutation. Attempts: %d\n", i+1)
			gameID, err = client.UpdateBroadcastSettings(oauth, channelID, gameName)
		}
	}
	require.NoError(t, err, "Unknown error occured. This endpoint isn't owned by DX. Please raise the attempts limit on retry or contact the team to fix.")
	assert.Equal(t, expectedGameID, gameID)
	return gameID
}

func setDynamicManagementFlag(t *testing.T, client gql.Client, oauth token.OAuth, expectedInstallationID, gameID string, isManaged bool) {
	input := gql.SetExtensionFeatureFlagsInput{
		InstallationID: expectedInstallationID,
		DynamicManagement: []gql.DynamicManagementInput{
			gql.DynamicManagementInput{
				GameID:    gameID,
				IsManaged: isManaged,
			},
		},
	}
	installedExtension, err := client.SetExtensionFeatureFlags(oauth, input)
	require.NoError(t, err)
	require.NotEmpty(t, installedExtension)
	require.NotEmpty(t, installedExtension.PermittedFeatures)
	require.NotNil(t, installedExtension.PermittedFeatures.DynamicManagement)
	pass := false
	for _, dm := range installedExtension.PermittedFeatures.DynamicManagement {
		if dm.Game.ID == gameID {
			assert.Equal(t, gameID, dm.Game.ID)
			pass = true
			break
		}
	}
	assert.True(t, pass)
}

func constructManifestID(id, version string) string {
	return fmt.Sprintf("%s:%s", id, version)
}

func constructInstallationID(id, version, channelID string) string {
	return fmt.Sprintf("%s:%s:%s", id, version, channelID)
}

func parseManifestID(id string) (string, string, error) {
	s, err := parseGQLID(id)
	if err != nil {
		return "", "", err
	}
	if len(s) != 2 {
		return "", "", errors.New("Invalid Manifest ID")
	}
	return s[0], s[1], nil
}

func parseInstallationID(id string) (string, string, string, error) {
	s, err := parseGQLID(id)
	if err != nil {
		return "", "", "", err
	}
	if len(s) != 3 {
		return "", "", "", errors.New("Invalid Installation ID")
	}
	return s[0], s[1], s[2], nil
}

func parseGQLID(id string) ([]string, error) {
	s := strings.Split(id, ":")
	if len(s) == 0 {
		return nil, errors.New("Error on parsing GQL ID")
	}
	return s, nil
}
