package gql_test

import (
	"fmt"
	"net/http"
	"testing"
	"time"

	"code.justin.tv/extensions/eastwatch/internal/api/gql"
	"code.justin.tv/extensions/eastwatch/internal/config"
	"code.justin.tv/extensions/eastwatch/internal/metrics"
	"code.justin.tv/extensions/eastwatch/internal/testutil"
	"code.justin.tv/extensions/eastwatch/internal/users"
	"github.com/stretchr/testify/assert"
	"github.com/stretchr/testify/require"
)

var (
	// Reference: https://git-aws.internal.justin.tv/gds/gds/blob/master/extensions/ems/dynamicmanagement/hardcoded_pairings.go
	extensionRecommendationsSample = map[string][]string{
		"493570": { // Clone Drone in the Danger Zone
			"xm9qmvnmls6ifd21kjejl0b30gyheo",
		},
	}
)

func TestGetExtensionRecommendations(t *testing.T) {
	errMsg := "Extension recommendations map might have been changed. Please refer to https://git-aws.internal.justin.tv/gds/gds/blob/master/extensions/ems/dynamicmanagement/hardcoded_pairings.go and update the test assertion. Notice: Extensions on the map may not have record in staging. Please call ems.GetExtension() or gql.GetExtension mutation to vertify."

	cfg := config.Staging
	bs, err := metrics.InitBootStrap("Eastwatch")
	if err != nil {
		panic(fmt.Sprintf("unable to initialize stats! %v", err))
	}
	captured := testutil.NewRoundTripRecorder(
		testutil.NewMetricsRoundTripper(http.DefaultTransport, bs))
	httpClient := &http.Client{
		Timeout:   2 * time.Second,
		Transport: captured,
	}

	client, err := gql.NewClient(httpClient, cfg.GQLURL)
	require.NoError(t, err)

	t.Run("with no extension recommendations", func(t *testing.T) {
		oauth, err := users.User1.OAuthToken(cfg, config.SiteTwilight)
		require.NoError(t, err)
		documents, err := client.GetExtensionRecommendations(oauth, nonMatchedGame.ID)
		require.NoError(t, err)
		require.NotNil(t, documents)
		assert.Equal(t, 0, len(documents))
	})

	t.Run("with extension recommendations", func(t *testing.T) {
		expectedState := "RELEASED"
		for gameID, expectedIDs := range extensionRecommendationsSample {
			oauth, err := users.User1.OAuthToken(cfg, config.SiteTwilight)
			require.NoError(t, err)
			documents, err := client.GetExtensionRecommendations(oauth, gameID)
			require.NoError(t, err)
			require.NotNil(t, documents)
			assert.Equal(t, 1, len(documents), fmt.Sprintf("%s\n", errMsg))
			for i, document := range documents {
				assert.Equal(t, expectedIDs[i], document.ID, fmt.Sprintf("%s\n", errMsg))
				assert.Equal(t, expectedState, document.State, fmt.Sprintf("%s\n", errMsg))
			}
		}
	})

	bs.SampleReporter.Stop()
}
