package gql_test

import (
	"fmt"
	"testing"

	"code.justin.tv/extensions/eastwatch/internal/api/gql"
	"github.com/stretchr/testify/assert"
	"github.com/stretchr/testify/require"
)

func TestSetExtensionFeatureFlags(t *testing.T) {
	t.Skip("[EXDSC-1110] test is brittle, causing Eastwatch to fail often")
	errMsg := "Content-Matched extension map might have been changed. Please refer to https://git-aws.internal.justin.tv/gds/gds/blob/master/extensions/ems/dynamicmanagement/hardcoded_pairings.go and update the test assertion. Notice: Extensions on the map may not have record in staging. Please call ems.GetExtension() or gql.GetExtension mutation to vertify."

	bs, client, userID, oauth := initiate(t)
	expectedInstallationID, expectedContentMatchedGames := getExpectedExtension(t, client, oauth, contentMatchedExtensionIDs[0], userID, true)
	expectedID, expectedVersion, _, err := parseInstallationID(expectedInstallationID)
	require.NoError(t, err)

	t.Run("failed", func(t *testing.T) {
		t.Run("extension does not exist", func(t *testing.T) {
			installationID := "extension_not_exist" + ":" + "0.0.1" + ":" + userID
			input := gql.SetExtensionFeatureFlagsInput{
				InstallationID: installationID,
				CanSentChat:    true,
				CanUseBits:     false,
			}

			uninstallExtension(t, client, oauth, installationID)
			ext, err := client.SetExtensionFeatureFlags(oauth, input)
			require.Error(t, err)
			assert.Empty(t, ext)
		})

		t.Run("extension is not installed", func(t *testing.T) {
			input := gql.SetExtensionFeatureFlagsInput{
				InstallationID: expectedInstallationID,
				CanSentChat:    true,
				CanUseBits:     false,
			}

			uninstallExtension(t, client, oauth, expectedInstallationID)
			ext, err := client.SetExtensionFeatureFlags(oauth, input)
			require.Error(t, err)
			assert.Empty(t, ext)
			setErr, ok := err.(*gql.SetExtensionFeatureFlagsError)
			assert.True(t, ok)
			assert.NotNil(t, setErr)
			assert.Equal(t, gql.ErrCodeExtensionNotInstalled, setErr.Code)
			assert.Equal(t, expectedInstallationID, setErr.MissingInstallationID)
		})
	})

	t.Run("success", func(t *testing.T) {
		t.Run("without dynamic management", func(t *testing.T) {
			input := gql.SetExtensionFeatureFlagsInput{
				InstallationID: expectedInstallationID,
				CanSentChat:    true,
				CanUseBits:     false,
			}

			installExtension(t, client, oauth, expectedInstallationID)
			defer uninstallExtension(t, client, oauth, expectedInstallationID)

			ext, err := client.SetExtensionFeatureFlags(oauth, input)
			assert.NoError(t, err)
			assert.NotEmpty(t, ext)
			assert.Equal(t, expectedID, ext.Extension.ExtensionID)
			assert.Equal(t, expectedVersion, ext.Extension.Version)
		})

		t.Run("with dynamic management, without content-matched game", func(t *testing.T) {
			input := gql.SetExtensionFeatureFlagsInput{
				InstallationID: expectedInstallationID,
				CanSentChat:    false,
				CanUseBits:     true,
				DynamicManagement: []gql.DynamicManagementInput{
					gql.DynamicManagementInput{
						GameID:    nonMatchedGame.ID,
						IsManaged: true,
					},
				},
			}

			installExtension(t, client, oauth, expectedInstallationID)
			defer uninstallExtension(t, client, oauth, expectedInstallationID)

			ext, err := client.SetExtensionFeatureFlags(oauth, input)
			assert.NoError(t, err, fmt.Sprintf(errMsg, expectedID, expectedVersion, userID))
			assert.NotEmpty(t, ext)
			assert.Equal(t, expectedID, ext.Extension.ExtensionID)
			assert.Equal(t, expectedVersion, ext.Extension.Version)
			assert.Nil(t, ext.PermittedFeatures.DynamicManagement)
			assert.True(t, ext.PermittedFeatures.CanUseBits)
			assert.False(t, ext.PermittedFeatures.CanSendChat)
		})

		t.Run("with dynamic management and content-matched game", func(t *testing.T) {
			input := gql.SetExtensionFeatureFlagsInput{
				InstallationID: expectedInstallationID,
				CanSentChat:    false,
				CanUseBits:     true,
				DynamicManagement: []gql.DynamicManagementInput{
					gql.DynamicManagementInput{
						GameID:    expectedContentMatchedGames[0].ID,
						IsManaged: true,
					},
				},
			}

			installExtension(t, client, oauth, expectedInstallationID)
			defer uninstallExtension(t, client, oauth, expectedInstallationID)

			ext, err := client.SetExtensionFeatureFlags(oauth, input)
			assert.NoError(t, err, fmt.Sprintf(errMsg, expectedID, expectedVersion, userID))
			assert.NotEmpty(t, ext)
			assert.NotNil(t, ext.PermittedFeatures.DynamicManagement, errMsg)
			assert.Equal(t, 1, len(ext.PermittedFeatures.DynamicManagement))
			assert.Equal(t, expectedContentMatchedGames[0].ID, ext.PermittedFeatures.DynamicManagement[0].Game.ID, errMsg)
			assert.True(t, ext.PermittedFeatures.DynamicManagement[0].IsManaged)
		})
	})

	bs.SampleReporter.Stop()
}
