package gql_test

import (
	"fmt"
	"net/http"
	"strings"
	"testing"
	"time"

	"github.com/stretchr/testify/assert"

	"code.justin.tv/extensions/eastwatch/internal/api/gql"
	"code.justin.tv/extensions/eastwatch/internal/config"
	"code.justin.tv/extensions/eastwatch/internal/metrics"
	"code.justin.tv/extensions/eastwatch/internal/testutil"
	"code.justin.tv/extensions/eastwatch/internal/users"
	"github.com/stretchr/testify/require"
)

func TestSearchExtensions(t *testing.T) {
	cfg := config.Staging
	bs, err := metrics.InitBootStrap("Eastwatch")
	if err != nil {
		panic(fmt.Sprintf("unable to initialize stats! %v", err))
	}
	captured := testutil.NewRoundTripRecorder(
		testutil.NewMetricsRoundTripper(http.DefaultTransport, bs))
	httpClient := &http.Client{
		Timeout:   2 * time.Second,
		Transport: captured,
	}

	client, err := gql.NewClient(httpClient, cfg.GQLURL)
	require.NoError(t, err)

	t.Run("With view_all_extensions capability", func(t *testing.T) {
		searchTerm := "league"
		maximumResult := 3
		expectedCount := 3
		expectedHit := 32

		// This user has been added to the view_all_extension whitelist
		oauth, err := users.User1.OAuthToken(cfg, config.SiteTwilight)
		require.NoError(t, err)
		extensions, totalHit, err := client.SearchExtensions(oauth, maximumResult, "", "league")
		require.NoError(t, err)
		resultCount := len(extensions)

		assert.True(t, resultCount <= maximumResult, fmt.Sprintf("resultCount (%d) should be <= maximumResult (%d)", resultCount, maximumResult))
		assert.True(t, resultCount <= totalHit, fmt.Sprintf("resultCount (%d) should be <= totalHit (%d)", resultCount, totalHit))
		assert.True(t, totalHit == expectedHit, fmt.Sprintf("Search test gets %d totalhit. Expected to be %d. Reason could be:\n1. Staging ES has been remirroring and data with the term '%s' have been changed. Please update the searchTerm in the code to fetch lastest data and update expectedHit;\n2. The user (%s) has been removed from the view_all_extension whitelist by mistake. Please add it back (see https://wiki.twitch.com/display/DS/Whitelist+Management).\n", totalHit, expectedHit, searchTerm, users.User1.ID()))
		assert.True(t, resultCount == expectedCount, fmt.Sprintf("Search test gets %d results. Expected to be %d. Reason could be:\n1. Staging ES has been remirroring and data with the term '%s' have been changed. Please update the searchTerm in the code to fetch lastest data and update expectedHit;\n2. The user (%s) has been removed from the view_all_extension whitelist by mistake.  Please add it back (see https://wiki.twitch.com/display/DS/Whitelist+Management).\n", resultCount, expectedCount, searchTerm, users.User1.ID()))

		for _, v := range extensions {
			// cast all to upper case as elasticsearch ignores capitalization
			// The way that gql would call with searchTerm: https://git-aws.internal.justin.tv/edge/graphql/blob/master/resolvers/queries/extension_connection.go#L132
			metSearchTerm := strings.Contains(strings.ToUpper(v.Name), strings.ToUpper(searchTerm)) ||
				strings.Contains(strings.ToUpper(v.Description), strings.ToUpper(searchTerm)) ||
				strings.Contains(strings.ToUpper(v.Summary), strings.ToUpper(searchTerm))

			assert.True(t, metSearchTerm, fmt.Sprintf("Search Term (%s) wasn't found in extension:\n%s:%s : %s %s %s", searchTerm, v.ID, v.Version, v.Name, v.Summary, v.Description))
		}
	})

	t.Run("With whitelist and without view_all_extensions capability", func(t *testing.T) {
		searchTerm := "league"
		maximumResult := 9
		expectedCount := 8 // maximum total hit is 8 - every released extension plus a whitelist one
		expectedHit := 8

		oauth, err := users.User2.OAuthToken(cfg, config.SiteTwilight)
		require.NoError(t, err)
		extensions, totalHit, err := client.SearchExtensions(oauth, maximumResult, "", "league")
		require.NoError(t, err)
		resultCount := len(extensions)

		assert.True(t, resultCount <= maximumResult, fmt.Sprintf("resultCount (%d) should be <= maximumResult (%d)", resultCount, maximumResult))
		assert.True(t, resultCount <= totalHit, fmt.Sprintf("resultCount (%d) should be <= totalHit (%d)", resultCount, totalHit))
		assert.True(t, totalHit == expectedHit, fmt.Sprintf("Search test gets %d totalhit. Expected to be %d. Staging ES might has been remirroring and data with the term '%s' might have been changed. Please update the searchTerm in the code to fetch lastest data and update expectedHit.\n", totalHit, expectedHit, searchTerm))
		assert.True(t, resultCount == expectedCount, fmt.Sprintf("Search test gets %d results. Expected to be %d. Staging ES might has been remirroring and data with the term '%s' might have been changed. Please update the searchTerm in the code to fetch lastest data and update expectedCount.\n", resultCount, expectedCount, searchTerm))

		foundWhitelist := false
		ext_name := "League for eastwatch"          // extension that has user2 in its whitelist
		ext_id := "justforeastwatchjustforeastwatc" // id for ext_name extension
		version_table := "extensions_extension_versions"
		for _, v := range extensions {
			// cast all to upper case as elasticsearch ignores capitalization
			// The way that gql would call with searchTerm: https://git-aws.internal.justin.tv/edge/graphql/blob/master/resolvers/queries/extension_connection.go#L132
			metSearchTerm := strings.Contains(strings.ToUpper(v.Name), strings.ToUpper(searchTerm)) ||
				strings.Contains(strings.ToUpper(v.Description), strings.ToUpper(searchTerm)) ||
				strings.Contains(strings.ToUpper(v.Summary), strings.ToUpper(searchTerm))

			// Name of eastwatch extension: "League for eastwatch"
			// the "_id" and extension_id of this test extension in ES is "justforeastwatchjustforeastwatc".
			// It is added manually in dynamoDB and ES seperately so there might be a lack of consistency between the dynamoDB and ES for this record.
			if strings.Contains(strings.ToUpper(v.Name), strings.ToUpper(ext_name)) {
				foundWhitelist = true
			}
			assert.True(t, metSearchTerm, fmt.Sprintf("Search Term (%s) wasn't found in extension:\n%s:%s : %s %s %s", searchTerm, v.ID, v.Version, v.Name, v.Summary, v.Description))
		}

		assert.True(t, foundWhitelist, fmt.Sprintf("Extension (%s) is expected to be in the searching results, since user (%s) has been added to its whitelist manually. To fix the problem, in STAGING:\n\t1. Manually add an extension named '%s' with the extension id '%s' in dynamoDB, table '%s'. You can copy over the json of an existing extension, modify the name and id then paste as new.\n\t2. Manually create and elasticsearch document of the extension, with the same name and id, and with the ES _id as '%s' as well (no version included in the _id for this extension). Again, you can copy and modify the value or an existing document in ES then paste as a new one.\n", ext_name, users.User2.ID(), ext_name, ext_id, version_table, ext_id))
	})

	t.Run("Without whitelist and view_all_extensions capability", func(t *testing.T) {
		searchTerm := "league"
		maximumResult := 9
		expectedCount := 7 // maximum total hit is 7
		expectedHit := 7

		oauth, err := users.User3.OAuthToken(cfg, config.SiteTwilight)
		require.NoError(t, err)
		extensions, totalHit, err := client.SearchExtensions(oauth, maximumResult, "", "league")
		require.NoError(t, err)
		resultCount := len(extensions)

		assert.True(t, resultCount <= maximumResult, fmt.Sprintf("resultCount (%d) should be <= maximumResult (%d)", resultCount, maximumResult))
		assert.True(t, resultCount <= totalHit, fmt.Sprintf("resultCount (%d) should be <= totalHit (%d)", resultCount, totalHit))
		assert.True(t, totalHit == expectedHit, fmt.Sprintf("Search test gets %d totalhit. Expected to be %d. Staging ES might has been remirroring and data with the term '%s' might have been changed. Please update the searchTerm in the code to fetch lastest data and update expectedHit.\n", totalHit, expectedHit, searchTerm))
		assert.True(t, resultCount == expectedCount, fmt.Sprintf("Search test gets %d results. Expected to be %d. Staging ES might has been remirroring and data with the term '%s' might have been changed. Please update the searchTerm in the code to fetch lastest data and update expectedCount.\n", resultCount, expectedCount, searchTerm))

		for _, v := range extensions {
			// cast all to upper case as elasticsearch ignores capitalization
			// The way that gql would call with searchTerm: https://git-aws.internal.justin.tv/edge/graphql/blob/master/resolvers/queries/extension_connection.go#L132
			metSearchTerm := strings.Contains(strings.ToUpper(v.Name), strings.ToUpper(searchTerm)) ||
				strings.Contains(strings.ToUpper(v.Description), strings.ToUpper(searchTerm)) ||
				strings.Contains(strings.ToUpper(v.Summary), strings.ToUpper(searchTerm))

			assert.True(t, metSearchTerm, fmt.Sprintf("Search Term (%s) wasn't found in extension:\n%s:%s : %s %s %s", searchTerm, v.ID, v.Version, v.Name, v.Summary, v.Description))
		}
	})

	bs.SampleReporter.Stop()
}
