package gql_test

import (
	"code.justin.tv/extensions/eastwatch/internal/testutil"
	"errors"
	"fmt"
	"net/http"
	"path"
	"strings"
	"testing"
	"time"

	"code.justin.tv/extensions/eastwatch/internal/api/cartman"
	"code.justin.tv/extensions/eastwatch/internal/api/ems"
	"code.justin.tv/extensions/eastwatch/internal/api/gql"
	"code.justin.tv/extensions/eastwatch/internal/api/owl"
	"code.justin.tv/extensions/eastwatch/internal/api/visage"
	"code.justin.tv/extensions/eastwatch/internal/config"
	"code.justin.tv/extensions/eastwatch/internal/models/token"
	"code.justin.tv/extensions/eastwatch/internal/users"
	"code.justin.tv/extensions/eastwatch/test"
	"code.justin.tv/foundation/twitchclient"
	cartmanclient "code.justin.tv/web/cartman/client"
	owlclient "code.justin.tv/web/owl/client"

	"github.com/stretchr/testify/suite"
)

func TestExtensionDevWorkflowGQLSuite(t *testing.T) {
	suite.Run(t, new(ExtensionDevWorkflowSuite))
}

// ExtensionDevWorkflowSuite implements and tests a development workflow for creating and activating
// an extension on the developer's own channel via the graphql edge
type ExtensionDevWorkflowSuite struct {
	test.EastwatchSuite

	// Test Configuration
	Config        config.Configuration
	DevUser       users.User
	OAuthDevsite  token.OAuth
	OAuthTwilight token.OAuth

	ExtensionName    string
	RedirectURI      string
	ExtensionVersion string
	PanelSlot        gql.PanelSlot

	// Clients
	GQL gql.Client

	// Responses
	//ExtensionClient   gql.ExtensionClient
	CurrentUser       gql.User
	ExtensionClient   string
	UserChannel       gql.UserChannel
	ExtensionManifest gql.ExtensionManifest

	InstalledExtension                 gql.InstalledExtension
	UninstalledExtensionInstallationID string

	CreatedPanel gql.Panel
	DeletedPanel gql.Panel

	AfterActivationInstalledExtensions   []gql.InstalledExtension
	AfterDeactivationInstalledExtensions []gql.InstalledExtension

	HardDeleteError error
}

func (s *ExtensionDevWorkflowSuite) Configure() {
	s.Config = config.Staging
	s.DevUser = users.User3
	s.ExtensionName = fmt.Sprintf("eastwatch-gql-%d", time.Now().Unix())
	s.ExtensionVersion = "0.1.2"
	s.RedirectURI = "https://localhost:8123"
	s.PanelSlot = gql.PanelSlotExtension1
	oad, err := s.DevUser.OAuthToken(s.Config, config.SiteDevsite)
	s.Require().NoError(err)
	s.OAuthDevsite = oad
	oat, err := s.DevUser.OAuthToken(s.Config, config.SiteTwilight)
	s.Require().NoError(err)
	s.OAuthTwilight = oat
	client, err := gql.NewClient(s.HTTPClient, s.Config.GQLURL)
	s.Require().NoError(err)
	s.GQL = client
	// Ensure user's channels are empty before we run tests
	s.DevUser.CleanUp(s.Config, s.HTTPClient, s.GQL)
}

// SetupSuite is run automatically by the testify suite
func (s *ExtensionDevWorkflowSuite) SetupSuite() {
	s.EastwatchSuite.SetupSuite()
	s.Configure()
	s.RunSetups(
		s.QueryCurrentUser,                     // gets the userid and displayname
		s.CreateExtensionClient,                // creates the client-id for the extension
		s.CreateExtensionManifest,              // initial manifest setup with testing for defaults
		s.MutationSaveExtensionManifest,        // updates the extension manifest details
		s.MutationSaveDiscoveryData,            // updates the extension discovery data
		s.CheckManifestErrors,                  // test if manifest errors when expected to
		s.MutationInstallExtension,             // install the extension onto DevUser's channel
		s.MutationCreatePanel,                  // creates an panel on DevUser's channel
		s.MutationApplyExtensionActivations,    // activates the extension on the panel create above
		s.QueryExtensionsForChannelCurrentUser, // verify extension exists and is active
	)
	s.RunTearDowns()
}

// ====================
// Setup Steps
// --------------------

func (s *ExtensionDevWorkflowSuite) QueryCurrentUser() error {

	// Setup
	user, err := s.GQL.GetCurrentUser(s.OAuthTwilight)
	if err != nil {
		return err
	}
	s.CurrentUser = user

	return nil
}

func (s *ExtensionDevWorkflowSuite) CreateExtensionClient() error {
	oc, err := owlclient.NewClient(twitchclient.ClientConf{
		Host: s.Config.OwlURL,
		RoundTripperWrappers: []func(http.RoundTripper) http.RoundTripper {
			testutil.TwitchClientTestHeaders,
		},
	})
	if err != nil {
		return fmt.Errorf("error creating owl client: %v", err)
	}
	owlClient := owl.NewClient(oc)

	extensionID, err := owlClient.CreateExtensionClient(s.CurrentUser.ID, s.ExtensionName)
	if err != nil {
		fmt.Printf("Could not create Extension Client for user %s: %s\n", s.CurrentUser.DisplayName, err)
		return err
	}

	s.ExtensionClient = extensionID

	s.AddTearDownStep(
		func() error {

			// gql delete extension performs a soft delete, as it would on devsite
			err := s.GQL.DeleteExtension(s.OAuthDevsite, s.ExtensionClient)
			if err != nil {
				fmt.Printf("GQL DeleteExtension got err: %s\n", err)
				return err
			}

			// hard delete is an admin endpoint in ems, capture any error separately from tear down
			// errors so that we can have a separate specific assertion for it
			s.HardDeleteError = s.HardDeleteExtension()

			err = owlClient.DeleteClient(s.CurrentUser.ID, s.ExtensionClient)
			if err != nil {
				fmt.Printf("Owl DeleteClient got err: %s\n", err)
			}
			return err
		})
	return nil
}

func (s *ExtensionDevWorkflowSuite) CreateExtensionClientOld() error {
	// due to peering and other connectivity issues, we're using Visage to generate the Extension ID
	// the GQL version (below) is still here in case things change and we want to revive it later.
	vc, err := visage.NewClient(s.HTTPClient, s.Config.VisageURL)
	if err != nil {
		return err
	}

	eid, err := vc.CreateExtensionID(s.OAuthDevsite, s.ExtensionName)
	if err != nil {
		return err
	}

	s.ExtensionClient = eid

	// TearDown
	s.AddTearDownStep(
		func() error {

			// gql delete extension performs a soft delete, as it would on devsite
			err := s.GQL.DeleteExtension(s.OAuthDevsite, s.ExtensionClient)

			// hard delete is an admin endpoint in ems, capture any error separately from tear down
			// errors so that we can have a separate specific assertion for it
			s.HardDeleteError = s.HardDeleteExtension()

			return err
		})

	return nil

}

/*
// Leaving this GraphQL CreateExtensionClientID in case we move everything to Owl staging
func (s *ExtensionDevWorkflowSuite) MutationCreateExtensionClient() error {

	// Setup
	client, err := s.GQL.CreateExtensionClient(s.OAuthDevsite, s.ExtensionName, s.RedirectURI)
	if err != nil {
		return err
	}
	s.ExtensionClient = client

	// TearDown
	s.AddTearDownStep(
		func() error {

			// gql delete extension performs a soft delete, as it would on devsite
			err := s.GQL.DeleteExtension(s.OAuthDevsite, s.ExtensionClient)

			// hard delete is an admin endpoint in ems, capture any error separately from tear down
			// errors so that we can have a separate specific assertion for it
			s.HardDeleteError = s.HardDeleteExtension()

			return err
		})

	return nil
}
*/

func (s *ExtensionDevWorkflowSuite) HardDeleteExtension() error {

	// Clients required for hard delete
	oc, err := owlclient.NewClient(twitchclient.ClientConf {
		Host: s.Config.OwlURL,
		RoundTripperWrappers: []func(http.RoundTripper) http.RoundTripper {
			testutil.TwitchClientTestHeaders,
		},
	})
	if err != nil {
		return fmt.Errorf("error creating owl client: %v", err)
	}
	owlClient := owl.NewClient(oc)

	cc, err := cartmanclient.NewClient(twitchclient.ClientConf{Host: s.Config.CartmanURL})
	if err != nil {
		return fmt.Errorf("error creating cartman client: %v", err)
	}
	cartmanClient := cartman.NewClient(cc)

	emsClient, err := ems.NewClient(s.HTTPClient, s.Config.EMSURL)
	if err != nil {
		return fmt.Errorf("error creating ems client: %v", err)
	}

	// Obtain AppAccess OAuth Token
	secret, err := config.GetSandstormSecret(s.Config.SandstormARN, path.Join(s.Config.BaseSandstormPath, s.Config.AppClientSecret))
	if err != nil {
		return fmt.Errorf("error getting app client secret from sandstorm: %v", err)
	}

	appAccessToken, err := owlClient.GetAppAccessToken(s.Config.ClientID, secret)
	if err != nil {
		return fmt.Errorf("error getting app access token from owl: %v", err)
	}

	// Obtain JWT
	jwt, err := cartmanClient.GetToken(appAccessToken,
		[]string{
			cartman.CapExtensionsViewAll,
			cartman.CapExtensionsHardDelete,
		})
	if err != nil {
		return fmt.Errorf("error getting jwt from cartman: %v", err)
	}

	// Uninstall the extension before hard delete, otherwise it'll max out the installed extension limitation
	installationID := s.ExtensionClient + ":" + s.ExtensionVersion + ":" + s.CurrentUser.ID
	_, err = s.GQL.UninstallExtension(s.OAuthTwilight, installationID)
	if err != nil {
		return fmt.Errorf("error executing ems hard delete when uninstalling the extension: %v", err)
	}

	// Execute Hard Delete
	err = emsClient.HardDeleteExtension(jwt, s.ExtensionClient)
	if err != nil {
		return fmt.Errorf("error executing ems hard delete: %v", err)
	}

	return nil
}

func (s *ExtensionDevWorkflowSuite) minimumValidExtensionManifestInput() gql.ExtensionManifestInput {
	return gql.ExtensionManifestInput{
		ID:      s.ExtensionClient,
		Version: s.ExtensionVersion,
		AssetManifest: gql.ExtensionAssetManifestInput{
			AssetHash: "",
		},
		Capabilities: gql.ExtensionCapabilitiesInput{
			ConfigurationLocation: "NONE",
			HasBitsSupport:        true,
			BitsSupportLevel:      "OPTIONAL",
			Whitelists: gql.ExtensionWhitelistsInput{
				Broadcasters: []string{},
				ConfigURLs:   []string{},
				PanelURLs:    []string{},
				Testers:      []string{},
			},
		},
		DeveloperManifest: gql.ExtensionDeveloperManifestInput{
			AuthorEmail:    "",
			TestingBaseURI: "",
		},
		DiscoveryManifest: gql.ExtensionDiscoveryManifest{
			AuthorName: s.DevUser.Name(),
			Name:       s.ExtensionName,
			Categories: []string{},
			Games:      []string{},
		},
		Views: gql.ExtensionViewsInput{
			Config: &gql.ConfigViewInput{
				ViewerPath: "config.html",
			},
			Component: &gql.ComponentViewInput{
				ViewerPath:   "test.html",
				HasAutoscale: false,
				AspectRatioX: 2,
				AspectRatioY: 3,
				TargetHeight: 4000,
			},
			Panel: &gql.PanelViewInput{
				ViewerPath: "panel.html",
				Height:     300,
			},
		},
	}
}

func (s *ExtensionDevWorkflowSuite) minimumValidExtensionDiscoveryInput() gql.ExtensionDiscoveryManifestInput {
	return gql.ExtensionDiscoveryManifestInput{
		ID:      s.ExtensionClient,
		Version: s.ExtensionVersion,
		DiscoveryManifest: gql.ExtensionDiscoveryManifest{
			AuthorName: s.DevUser.Name(),
			Name:       s.ExtensionName,
			Categories: []string{},
			Games:      []string{},
		},
	}
}

func (s *ExtensionDevWorkflowSuite) CreateExtensionManifest() error {
	// Setup + SupportLevel default test
	input := s.minimumValidExtensionManifestInput()
	input.Capabilities.BitsSupportLevel = "REQUIRED"

	manifest, err := s.GQL.SaveExtensionManifest(s.OAuthDevsite, input)
	if err != nil {
		return err
	}

	if manifest.Capabilities.SubscriptionsSupportLevel != "NONE" {
		errCustom := errors.New("SubscriptionsSupportLevel not defaulting")
		return errCustom
	}
	if manifest.Capabilities.BitsSupportLevel != "REQUIRED" {
		errCustom := errors.New("BitsSupportLevel not set correctly")
		return errCustom
	}
	s.ExtensionManifest = manifest

	return nil
}

func (s *ExtensionDevWorkflowSuite) MutationSaveExtensionManifest() error {
	// Modification test of BitsSupportLevel and SubscriptionSupportLevel
	input := s.minimumValidExtensionManifestInput()
	sublevel := "OPTIONAL"
	input.Capabilities.SubscriptionsSupportLevel = &sublevel

	// Valid Input produces no errors
	manifest, err := s.GQL.SaveExtensionManifest(s.OAuthDevsite, input)
	if err != nil {
		return err
	}

	if manifest.Capabilities.SubscriptionsSupportLevel != "OPTIONAL" {
		errCustom := errors.New("SubscriptionsSupportLevel not set correctly")
		return errCustom
	}
	if manifest.Capabilities.BitsSupportLevel != "OPTIONAL" {
		errCustom := errors.New("BitsSupportLevel not set correctly")
		return errCustom
	}
	s.ExtensionManifest = manifest

	return nil
}

func (s *ExtensionDevWorkflowSuite) MutationSaveDiscoveryData() error {
	// Verify that summary and viewer summary stay different
	input := s.minimumValidExtensionDiscoveryInput()
	input.DiscoveryManifest.Summary = "A summary"
	input.DiscoveryManifest.ViewerSummary = "A viewer summary"
	input.DiscoveryManifest.ScreenshotURLs = []string{"https://example.com/screenshot.png"}

	// Valid Input produces no errors
	manifest, err := s.GQL.SaveDiscoveryData(s.OAuthDevsite, input)
	if err != nil {
		return err
	}

	if manifest.DiscoveryManifest.Summary != "A summary" {
		errCustom := errors.New("Summary not set correctly")
		return errCustom
	}
	if manifest.DiscoveryManifest.ViewerSummary != "A viewer summary" {
		errCustom := errors.New("ViewerSummary not set correctly")
		return errCustom
	}
	if len(manifest.DiscoveryManifest.ScreenshotURLs) != 0 {
		errCustom := errors.New("ScreenshotURLs set incorrectly")
		return errCustom
	}
	s.ExtensionManifest.DiscoveryManifest = manifest.DiscoveryManifest

	// TODO: Add a cleanup for deleting the discovery data once disco data is not in EMS

	return nil
}

func (s *ExtensionDevWorkflowSuite) CheckManifestErrors() error {
	errCustom := ""

	// Test Case: minimum valid input -> success
	success := s.minimumValidExtensionManifestInput()

	_, err := s.GQL.SaveExtensionManifest(s.OAuthDevsite, success)
	if err != nil {
		errCustom += "Expecetd Success, instead recieved:" + err.Error() + ";\n"
	}

	// Test Case: extension ID = "" -> ERROR: extension ID must be specified
	fail := s.minimumValidExtensionManifestInput()
	fail.ID = ""

	_, err = s.GQL.SaveExtensionManifest(s.OAuthDevsite, fail)

	if err == nil {
		errCustom += "Expected Error: \"extension ID must be specified\", instead recieved no error;\n"
	}
	if !strings.Contains(err.Error(), "extension ID must be specified") {
		errCustom += "Call not erroring with correct error message (extension ID must be specified) instead received " + err.Error() + ";\n"
	}

	// Test CASE:  Extension version = "" -> ERROR: extension version must be specified
	fail = s.minimumValidExtensionManifestInput()
	fail.Version = ""
	_, err = s.GQL.SaveExtensionManifest(s.OAuthDevsite, fail)

	if err == nil {
		errCustom += "Expected Error: \"extension version must be specified\", instead recieved no error;\n"
	}
	if !strings.Contains(err.Error(), "extension version must be specified") {
		errCustom += "Call not erroring with correct error message (extension version must be specified) instead received" + err.Error() + ";\n"
	}

	// Test Case: HasAutoscale = true; ScalePixels not provided -> INVALID_SCALE_PIXELS
	fail = s.minimumValidExtensionManifestInput()
	fail.Views.Component.HasAutoscale = true
	_, err = s.GQL.SaveExtensionManifest(s.OAuthDevsite, fail)

	if err == nil {
		errCustom += "Expected INVALID_SCALING_PIXELS, instead recieved no error\n"
	}
	if err != nil && err.Error() != "INVALID_SCALING_PIXELS" {
		errCustom += "Call not erroring with correct error message (INVALID_SCALING_PIXELS) instead received" + err.Error() + ";\n"
	}

	// Test Case: AspectRatioX = 0 -> INVALID_COMPONENT_ASPECT_RATIO_X
	fail = s.minimumValidExtensionManifestInput()
	fail.Views.Component.AspectRatioX = 0
	_, err = s.GQL.SaveExtensionManifest(s.OAuthDevsite, fail)

	if err == nil {
		errCustom += "Expected INVALID_COMPONENT_ASPECT_RATIO_X, instead recieved no error;\n"
	}
	if err == nil || err.Error() != "INVALID_COMPONENT_ASPECT_RATIO_X" {
		errCustom += "Call not erroring with correct error message (INVALID_COMPONENT_ASPECT_RATIO_X) instead received" + err.Error() + ";\n"
	}

	// Test Case: AspectRatioY = 0 -> INVALID_COMPONENT_ASPECT_RATIO_Y
	fail = s.minimumValidExtensionManifestInput()
	fail.Views.Component.AspectRatioY = 0
	_, err = s.GQL.SaveExtensionManifest(s.OAuthDevsite, fail)

	if err == nil {
		errCustom += "Expected INVALID_COMPONENT_ASPECT_RATIO_Y, instead recieved no error;\n"
	}
	if err == nil || err.Error() != "INVALID_COMPONENT_ASPECT_RATIO_Y" {
		errCustom += "Call not erroring with correct error message (INVALID_COMPONENT_ASPECT_RATIO_Y) instead received" + err.Error() + ";\n"
	}

	// Test Case: TaretHeight = 0 -> INVALID_COMPONENT_TARGET_HEIGHT
	fail = s.minimumValidExtensionManifestInput()
	fail.Views.Component.TargetHeight = 0
	_, err = s.GQL.SaveExtensionManifest(s.OAuthDevsite, fail)

	if err == nil {
		errCustom += "Expected INVALID_COMPONENT_TARGET_HEIGHT, instead recieved no error;\n"
	}
	if err == nil || err.Error() != "INVALID_COMPONENT_TARGET_HEIGHT" {
		errCustom += "Call not erroring with correct error message (INVALID_COMPONENT_TARGET_HEIGHT) instead received" + err.Error() + ";\n"
	}

	if errCustom != "" {
		errCustomAcc := errors.New(errCustom)
		return errCustomAcc
	}

	return nil
}

func (s *ExtensionDevWorkflowSuite) MutationInstallExtension() error {
	// Setup
	ext, err := s.GQL.InstallExtension(s.OAuthTwilight, s.ExtensionManifest.ManifestID, s.CurrentUser.ID)
	if err != nil {
		return err
	}
	s.InstalledExtension = ext

	// TearDown
	s.AddTearDownStep(func() error {
		extID, err := s.GQL.UninstallExtension(s.OAuthTwilight, s.InstalledExtension.InstallationID)
		if err != nil {
			return err
		}
		s.UninstalledExtensionInstallationID = extID

		return nil
	})

	return nil
}

func (s *ExtensionDevWorkflowSuite) MutationCreatePanel() error {

	// Setup
	panel, err := s.GQL.CreatePanel(s.OAuthTwilight, s.CurrentUser.ID, gql.ExtensionPanel)
	if err != nil {
		return err
	}
	s.CreatedPanel = panel

	// TearDown
	s.AddTearDownStep(func() error {
		panel, err := s.GQL.DeletePanel(s.OAuthTwilight, s.CreatedPanel.ID)
		if err != nil {
			return err
		}
		s.DeletedPanel = panel

		return nil
	})

	return nil
}

func (s *ExtensionDevWorkflowSuite) MutationApplyExtensionActivations() error {

	// Setup
	input := gql.ExtensionActivationsInput{
		InstallationID: s.InstalledExtension.InstallationID,
		Panel: &gql.PanelActivationInput{
			Slot: s.PanelSlot,
		},
	}

	exts, err := s.GQL.ApplyExtensionActivations(s.OAuthTwilight, s.CurrentUser.ID, input)
	if err != nil {
		return err
	}
	s.AfterActivationInstalledExtensions = exts

	// TearDown
	s.AddTearDownStep(func() error {
		input := gql.ExtensionActivationsInput{
			InstallationID: s.InstalledExtension.InstallationID,
			Panel:          nil,
		}

		exts, err := s.GQL.ApplyExtensionActivations(s.OAuthTwilight, s.CurrentUser.ID, input)
		if err != nil {
			return err
		}
		s.AfterDeactivationInstalledExtensions = exts

		if len(s.AfterDeactivationInstalledExtensions) != 1 {
			//Retry
			s.T().Logf("Repull list of extensions")
			//Adding buffer between calls to reduce 409 issues
			time.Sleep(time.Duration(250) * time.Millisecond)
			exts, err := s.GQL.ApplyExtensionActivations(s.OAuthTwilight, s.CurrentUser.ID, input)
			if err != nil {
				return err
			}
			s.AfterDeactivationInstalledExtensions = exts
		}

		return nil
	})

	return nil
}

func (s *ExtensionDevWorkflowSuite) QueryExtensionsForChannelCurrentUser() error {

	// Setup
	r, err := s.GQL.GetExtensionsForChannelCurrentUser(s.OAuthTwilight, s.CurrentUser.ID)
	if err != nil {
		return err
	}
	s.UserChannel = r

	return nil
}

// ====================
// Assertions
// --------------------

func (s *ExtensionDevWorkflowSuite) TestWorkflow() {

	// make sure the extension is uninstalled even if there's any error in the middle
	if s.OAuthTwilight != "" && s.InstalledExtension.InstallationID != "" {
		defer s.GQL.UninstallExtension(s.OAuthTwilight, s.InstalledExtension.InstallationID)
	}

	s.Require().NoError(s.SetupError, "setup failed to complete")

	s.Run("should return the correct user information", func() {
		actual := s.CurrentUser
		expected := gql.User{
			ID:                  s.DevUser.ID(),
			DisplayName:         s.DevUser.Name(),
			InstalledExtensions: []gql.InstalledExtension{},
		}
		s.Equal(expected, actual)
	})

	s.Run("should return an extension client id", func() {

		actual := s.ExtensionClient
		s.NotEmpty(actual)
		//s.Equal(s.ExtensionName, actual.Name)
		//s.Equal(s.RedirectURI, actual.RedirectURI)
		//s.Equal(time.Now().YearDay(), actual.CreatedAt.YearDay())
	})

	s.Run("should update the extension manifest", func() {

		actual := s.ExtensionManifest
		expected := gql.ExtensionManifest{
			ManifestID:   fmt.Sprintf("%s:%s", s.ExtensionClient, s.ExtensionVersion),
			Version:      s.ExtensionVersion,
			Capabilities: s.ExtensionManifest.Capabilities,
			DiscoveryManifest: gql.ExtensionDiscoveryManifest{
				AuthorName:     s.CurrentUser.DisplayName,
				Name:           s.ExtensionName,
				ScreenshotURLs: []string{},
				Summary:        "A summary",
				ViewerSummary:  "A viewer summary",
				Games:          []string{},
				Categories:     []string{},
			},
		}
		s.Equal(expected, actual)
	})

	s.Run("should install the extension onto the dev user channel", func() {
		actual := s.InstalledExtension
		expected := gql.InstalledExtension{
			InstallationID: fmt.Sprintf("%s:%s:%s", s.ExtensionClient, s.ExtensionVersion, s.CurrentUser.ID),
			Extension: gql.ExtensionVersion{
				ManifestID:  fmt.Sprintf("%s:%s", s.ExtensionClient, s.ExtensionVersion),
				ExtensionID: s.ExtensionClient,
				Name:        s.ExtensionName,
				Version:     s.ExtensionVersion,
			},
			ActivationConfig: gql.ActivationConfig{
				Anchor: "",
				Slot:   "",
				State:  "INACTIVE",
			},
		}
		s.Equal(expected, actual)
	})

	s.Run("should create an extension panel", func() {
		s.NotEmpty(s.CreatedPanel.ID)
		s.Equal(gql.ExtensionPanel, s.CreatedPanel.Type)
	})

	s.Run("should activate the extension onto the panel", func() {

		s.Require().Len(s.AfterActivationInstalledExtensions, 1)

		actual := s.AfterActivationInstalledExtensions[0]
		expected := gql.InstalledExtension{
			InstallationID: s.InstalledExtension.InstallationID,
			Extension:      s.InstalledExtension.Extension,
			ActivationConfig: gql.ActivationConfig{
				Anchor: "PANEL",
				Slot:   s.PanelSlot,
				State:  "ACTIVE",
			},
		}
		s.Equal(expected, actual)
	})

	s.Run("should see that the current user's channel has the extension installed and active", func() {

		actual := s.UserChannel
		expected := gql.UserChannel{
			CurrentUser: gql.User{
				ID:                  s.CurrentUser.ID,
				DisplayName:         s.CurrentUser.DisplayName,
				InstalledExtensions: nil,
			},
			Channel: gql.Channel{
				ID: s.CurrentUser.ID,
				InstalledExtensions: []gql.InstalledExtension{
					{
						InstallationID: s.InstalledExtension.InstallationID,
						Extension:      s.InstalledExtension.Extension,
						ActivationConfig: gql.ActivationConfig{
							Anchor: "PANEL",
							Slot:   s.PanelSlot,
							State:  "ACTIVE",
						},
					},
				},
			},
		}
		s.Equal(expected, actual)
	})

	s.Run("should deactivate the extension", func() {
		s.Require().Len(s.AfterDeactivationInstalledExtensions, 1)

		actual := s.AfterDeactivationInstalledExtensions[0]
		expected := gql.InstalledExtension{
			InstallationID: s.InstalledExtension.InstallationID,
			Extension:      s.InstalledExtension.Extension,
			ActivationConfig: gql.ActivationConfig{
				Anchor: "",
				Slot:   "",
				State:  "INACTIVE",
			},
		}
		s.Equal(expected, actual)
	})

	s.Run("should delete the panel", func() {
		// graphql mutation just returns back the same panel info with no error
		s.Equal(s.CreatedPanel, s.DeletedPanel)
	})

	s.Run("should uninstall the extension", func() {

		s.Equal(s.InstalledExtension.InstallationID, s.UninstalledExtensionInstallationID)
	})

	s.Run("should not have any tear down errors", func() {

		s.Empty(s.TearDownErrors)
	})

	s.Run("should hard delete extension", func() {

		s.NoError(s.HardDeleteError)
	})
}


