package loggers

import (
	"encoding/json"
	"fmt"
)

const (
	UnmappedKey = "UNKEYED_VALUE"
	MsgKey      = "msg"
)

// JSONLogger is a Logger that outputs to JSON.
// All keys and values are converted into strings via fmt.Sprint(). These keyvals are then used to construct a map of
// key->value which is marshaled into JSON using Encode on the JSONLogger's json.Encoder. If the encoding process fails
// (returns an error), and a backup ErrorHandler exists, it is used (the ErrorHandler's Log method is called).
// This serves as a backup logger which is sent the msg and keyvals that failed encoding along with the returned error.
type JSONLogger struct {
	// The json.Encoder to use for marshalling the map into JSON
	Dest *json.Encoder
	// The ErrorHandler to use as a backup logger if the dest fails to encode the content
	OnError ErrorHandler
}

// Log will take a msg and optional keyvals and log the content using the JSONLogger's json.Encoder
func (j *JSONLogger) Log(msg string, keyvals ...interface{}) {
	m := mapFromKeyVals(msg, keyvals...)
	err := j.Dest.Encode(m)
	// Fall back to the ErrorHandler if available
	if err != nil && j.OnError != nil {
		j.OnError.ErrorLogger(err).Log(msg, keyvals...)
	}
}

func mapFromKeyVals(msg string, keyvals ...interface{}) map[string]string {
	// Determine the length of the map
	// The size of the map we need is half the size of keyvals or keyvals + 1 (if keyvals is odd to account for
	// the odd, unmapped value). We then add +1 to the keyval-based result to account for actual message being logged
	mSize := ((len(keyvals) + (len(keyvals) % 2)) / 2) + 1
	m := make(map[string]string, mSize)
	// Add all the key, value pairs
	for i := 0; i < len(keyvals); i += 2 {
		var key, value interface{}
		if i == len(keyvals)-1 {
			key, value = UnmappedKey, keyvals[i]
		} else {
			key, value = keyvals[i], keyvals[i+1]
		}
		m[fmt.Sprint(key)] = fmt.Sprint(value)
	}
	// Add the final message
	m[MsgKey] = msg
	return m
}
