package instrumentation

import (
	"context"
	"net/http"
	"time"

	"goji.io/middleware"
)

type ContextInstrumenter interface {
	Instrument(ctx context.Context, name string, status int, callTime time.Duration, rate float32)
}
type DynamicContextInstrumenter func(context.Context) []ContextInstrumenter

type InstrumentManger struct {
	dynamic DynamicContextInstrumenter
	rates   map[Frequency]float32
}

// NewInstrumentor creates an Instrumntor that generates stats based on the included frequencies
// func NewInstrumentor(r map[Frequency]float32, ins ...ContextInstrumenter) InstrumentManger {
// 	return InstrumentManger{ins, r}
// }
func NewInstrumentor(r map[Frequency]float32, dynamic DynamicContextInstrumenter) InstrumentManger {
	return InstrumentManger{dynamic, r}
}

func (s InstrumentManger) instrument(ctx context.Context, wait func() int, defaultName string, defaultFrequency Frequency) {
	start := time.Now()
	code := wait()

	callTime := time.Since(start)
	var route interface{}
	route = middleware.Handler(ctx)

	name := defaultName
	predicted := defaultFrequency
	if cast, ok := route.(InstrumentedHandler); ok {
		name = cast.Name()
		predicted = cast.Frequency()
	}
	rate, ok := s.rates[predicted]
	if !ok {
		rate = 1
	}
	contextIn := s.dynamic(ctx)

	for _, d := range contextIn {
		d.Instrument(ctx, name, code, callTime, rate)
	}
}

func (s InstrumentManger) Middleware(defaultName string, defaultFrequency Frequency) func(http.Handler) http.Handler {
	return func(inner http.Handler) http.Handler {
		return http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
			wr := &statusWriter{inner: w}
			adapter := NewEventAdapter()
			defer adapter.Complete(&wr.status)
			go s.instrument(r.Context(), adapter.Wait, defaultName, defaultFrequency)
			inner.ServeHTTP(wr, r)
		})
	}
}
