package auth

import (
	"context"
	"fmt"
	"reflect"
	"time"

	wp "code.justin.tv/gds/gds/extensions/whitelist/protocol"
	"github.com/afex/hystrix-go/hystrix"
	statsdClient "github.com/cactus/go-statsd-client/statsd"
	"github.com/sirupsen/logrus"

	barbradytwirp "code.justin.tv/amzn/TwitchExtensionsBarbradyTwirp"
)

type BarbradyCredentials struct {
	alreadyCompared     map[string]bool
	capabilities        []string
	client              barbradytwirp.TwitchExtensionsBarbrady
	endpoint            string
	fallbackCredentials Credentials
	grantedCapabilities map[string]*barbradytwirp.Capability
	parameters          map[string]string
	shouldCompare       bool
	statsd              statsdClient.Statter
	whitelistAdapter    *whitelistAdapter
}

func NewBarbradyCredentials(fallbackCredentials Credentials, barbradyClient barbradytwirp.TwitchExtensionsBarbrady, whitelistAdapter *whitelistAdapter, statsd statsdClient.Statter) Credentials {
	return &BarbradyCredentials{
		alreadyCompared:     map[string]bool{},
		fallbackCredentials: fallbackCredentials,
		client:              barbradyClient,
		grantedCapabilities: map[string]*barbradytwirp.Capability{},
		shouldCompare:       true,
		statsd:              statsd,
		whitelistAdapter:    whitelistAdapter,
	}
}

// Methods that have to be called on Cartman / GQL Token

func (b *BarbradyCredentials) ClientID() string {
	return b.fallbackCredentials.ClientID()
}

func (b *BarbradyCredentials) UserID() *string {
	return b.fallbackCredentials.UserID()
}

func (b *BarbradyCredentials) FlatUserID() string {
	return b.fallbackCredentials.FlatUserID()
}

func (b *BarbradyCredentials) HomeChannel() string {
	return b.fallbackCredentials.HomeChannel()
}

func (b *BarbradyCredentials) CanAssumeIdentity(userID string) bool {
	return b.fallbackCredentials.CanAssumeIdentity(userID)
}

func (b *BarbradyCredentials) CanSkipMobileCheck(extensionID string) bool {
	return b.fallbackCredentials.CanSkipMobileCheck(extensionID)
}

func (b *BarbradyCredentials) CanSeeHiddenExtensions() bool {
	barbradyGranted := b.BarbradyCanSeeHiddenExtensions()
	fallbackGranted := b.fallbackCredentials.CanSeeHiddenExtensions()

	b.compare("CanSeeHiddenExtensions", barbradyGranted, fallbackGranted)

	return fallbackGranted
}

func (b *BarbradyCredentials) BarbradyCanSeeHiddenExtensions() bool {
	return b.BarbradyCanModerateExtensions() ||
		b.BarbradyCanReviewExtensions() ||
		b.BarbradyCanEditAllCategories() ||
		b.BarbradyCanViewAllExtensions()
}

// Whitelist Capabilities

func (b *BarbradyCredentials) CanListWhitelists() bool {
	barbradyGranted := b.BarbradyCanListWhitelists()
	fallbackGranted := b.fallbackCredentials.CanListWhitelists()

	b.compare("CanListWhitelists", barbradyGranted, fallbackGranted)

	return fallbackGranted
}

func (b *BarbradyCredentials) BarbradyCanListWhitelists() bool {
	return b.isWhitelistMember(wp.ViewAllWhitelists)
}

func (b *BarbradyCredentials) CanViewWhitelist(action wp.Action) bool {
	barbradyGranted := b.BarbradyCanViewWhitelist(action)
	fallbackGranted := b.fallbackCredentials.CanViewWhitelist(action)

	b.compare("CanViewWhitelist", barbradyGranted, fallbackGranted)

	return fallbackGranted
}

func (b *BarbradyCredentials) BarbradyCanViewWhitelist(action wp.Action) bool {
	return b.isWhitelistViewer(action) || b.isWhitelistMember(wp.ViewAllWhitelists)
}

func (b *BarbradyCredentials) CanEditWhitelist(action wp.Action) bool {
	barbradyGranted := b.BarbradyCanEditWhitelist(action)
	fallbackGranted := b.fallbackCredentials.CanEditWhitelist(action)

	b.compare("CanEditWhitelist", barbradyGranted, fallbackGranted)

	return fallbackGranted
}

func (b *BarbradyCredentials) BarbradyCanEditWhitelist(action wp.Action) bool {
	return b.isWhitelistEditor(action) || b.isWhitelistMember(wp.EditAllWhitelists)
}

func (b *BarbradyCredentials) CanViewAllExtensions() bool {
	barbradyGranted := b.BarbradyCanViewAllExtensions()
	fallbackGranted := b.fallbackCredentials.CanViewAllExtensions()

	b.compare("CanViewAllExtensions", barbradyGranted, fallbackGranted)

	return fallbackGranted
}

func (b *BarbradyCredentials) BarbradyCanViewAllExtensions() bool {
	return b.isWhitelistMember(wp.ViewAllExtensions)
}

func (b *BarbradyCredentials) CanInstallAllExtensions() bool {
	barbradyGranted := b.BarbradyCanInstallAllExtensions()
	fallbackGranted := b.fallbackCredentials.CanInstallAllExtensions()

	b.compare("CanInstallAllExtensions", barbradyGranted, fallbackGranted)

	return fallbackGranted
}

func (b *BarbradyCredentials) BarbradyCanInstallAllExtensions() bool {
	return b.isWhitelistMember(wp.InstallAllExtensions)
}

func (b *BarbradyCredentials) CanReviewExtensions() bool {
	barbradyGranted := b.BarbradyCanReviewExtensions()
	fallbackGranted := b.fallbackCredentials.CanReviewExtensions()

	b.compare("CanReviewExtensions", barbradyGranted, fallbackGranted)

	return fallbackGranted
}

func (b *BarbradyCredentials) BarbradyCanReviewExtensions() bool {
	return b.isWhitelistMember(wp.ReviewExtensions)
}

func (b *BarbradyCredentials) CanModerateExtensions() bool {
	barbradyGranted := b.BarbradyCanModerateExtensions()
	fallbackGranted := b.fallbackCredentials.CanModerateExtensions()

	b.compare("CanModerateExtensions", barbradyGranted, fallbackGranted)

	return fallbackGranted
}

func (b *BarbradyCredentials) BarbradyCanModerateExtensions() bool {
	return b.isWhitelistMember(wp.ModerateExtensions)
}

func (b *BarbradyCredentials) CanReviveExtensions() bool {
	barbradyGranted := b.BarbradyCanReviveExtensions()
	fallbackGranted := b.fallbackCredentials.CanReviveExtensions()

	b.compare("CanReviveExtensions", barbradyGranted, fallbackGranted)

	return fallbackGranted
}

func (b *BarbradyCredentials) BarbradyCanReviveExtensions() bool {
	return b.isWhitelistMember(wp.ReviveExtensions)
}

func (b *BarbradyCredentials) CanCreateExtensions() bool {
	barbradyGranted := b.BarbradyCanCreateExtensions()
	fallbackGranted := b.fallbackCredentials.CanCreateExtensions()

	b.compare("CanCreateExtensions", barbradyGranted, fallbackGranted)

	return fallbackGranted
}

func (b *BarbradyCredentials) BarbradyCanCreateExtensions() bool {
	return b.isWhitelistMember(wp.CreateExtensions)
}

func (b *BarbradyCredentials) CanCurateAllCategories() bool {
	barbradyGranted := b.BarbradyCanCurateAllCategories()
	fallbackGranted := b.fallbackCredentials.CanCurateAllCategories()

	b.compare("CanCurateAllCategories", barbradyGranted, fallbackGranted)

	return fallbackGranted
}

func (b *BarbradyCredentials) BarbradyCanCurateAllCategories() bool {
	return b.isWhitelistMember(wp.CurateAllCategories)
}

func (b *BarbradyCredentials) CanEditAllCategories() bool {
	barbradyGranted := b.BarbradyCanEditAllCategories()
	fallbackGranted := b.fallbackCredentials.CanEditAllCategories()

	b.compare("CanEditAllCategories", barbradyGranted, fallbackGranted)

	return fallbackGranted
}

func (b *BarbradyCredentials) BarbradyCanEditAllCategories() bool {
	return b.isWhitelistMember(wp.EditAllCategories)
}

func (b *BarbradyCredentials) CanCurateAllGames() bool {
	barbradyGranted := b.BarbradyCanCurateAllGames()
	fallbackGranted := b.fallbackCredentials.CanCurateAllGames()

	b.compare("CanCurateAllGames", barbradyGranted, fallbackGranted)

	return fallbackGranted
}

func (b *BarbradyCredentials) BarbradyCanCurateAllGames() bool {
	return b.isWhitelistMember(wp.CurateAllCategories)
}

func (b *BarbradyCredentials) CanHardDeleteExtensions() bool {
	barbradyGranted := b.BarbradyCanHardDeleteExtensions()
	fallbackGranted := b.fallbackCredentials.CanHardDeleteExtensions()

	b.compare("CanHardDeleteExtensions", barbradyGranted, fallbackGranted)

	return fallbackGranted
}

func (b *BarbradyCredentials) BarbradyCanHardDeleteExtensions() bool {
	return b.isWhitelistMember(wp.HardDeleteExtensions)
}

// Capabilities from Barbrady

func (b *BarbradyCredentials) CanInstallInto(channelID string) bool {
	barbradyGranted := b.BarbradyCanInstallInto(channelID)
	fallbackGranted := b.fallbackCredentials.CanInstallInto(channelID)

	b.compare("CanInstallInto", barbradyGranted, fallbackGranted)

	return fallbackGranted
}

func (b *BarbradyCredentials) BarbradyCanInstallInto(channelID string) bool {
	return b.checkGrantedCapabilities(CapEditInstalls, &map[string]string{ParamChannelID: channelID})
}

func (b *BarbradyCredentials) CanActivateOn(channelID string) bool {
	barbradyGranted := b.BarbradyCanActivateOn(channelID)
	fallbackGranted := b.fallbackCredentials.CanActivateOn(channelID)

	b.compare("CanActivateOn", barbradyGranted, fallbackGranted)

	return fallbackGranted
}

func (b *BarbradyCredentials) BarbradyCanActivateOn(channelID string) bool {
	return b.checkGrantedCapabilities(CapEditActivations, &map[string]string{ParamChannelID: channelID})
}

func (b *BarbradyCredentials) CanCreateVersion(extensionID string) bool {
	barbradyGranted := b.BarbradyCanCreateVersion(extensionID)
	fallbackGranted := b.fallbackCredentials.CanCreateVersion(extensionID)

	b.compare("CanCreateVersion", barbradyGranted, fallbackGranted)

	return fallbackGranted
}

func (b *BarbradyCredentials) BarbradyCanCreateVersion(extensionID string) bool {
	return b.checkGrantedCapabilities(CapCreateVersion, &map[string]string{ParamExtensionID: extensionID})
}

func (b *BarbradyCredentials) CanReadDeveloperOnlyData(extensionID string) bool {
	barbradyGranted := b.BarbradyCanReadDeveloperOnlyData(extensionID)
	fallbackGranted := b.fallbackCredentials.CanReadDeveloperOnlyData(extensionID)

	b.compare("CanReadDeveloperOnlyData", barbradyGranted, fallbackGranted)

	return fallbackGranted
}

func (b *BarbradyCredentials) BarbradyCanReadDeveloperOnlyData(extensionID string) bool {
	return b.checkGrantedCapabilities(CapCreateVersion, &map[string]string{ParamExtensionID: extensionID})
}

func (b *BarbradyCredentials) CanEditVersion(extensionID string) bool {
	barbradyGranted := b.BarbradyCanEditVersion(extensionID)
	fallbackGranted := b.fallbackCredentials.CanEditVersion(extensionID)

	b.compare("CanEditVersion", barbradyGranted, fallbackGranted)

	return fallbackGranted
}

func (b *BarbradyCredentials) BarbradyCanEditVersion(extensionID string) bool {
	return b.checkGrantedCapabilities(CapEditVersion, &map[string]string{ParamExtensionID: extensionID})
}

func (b *BarbradyCredentials) CanValidateInstall(extensionID string) bool {
	barbradyGranted := b.BarbradyCanValidateInstall(extensionID)
	fallbackGranted := b.fallbackCredentials.CanValidateInstall(extensionID)

	b.compare("CanValidateInstall", barbradyGranted, fallbackGranted)

	return fallbackGranted
}

func (b *BarbradyCredentials) BarbradyCanValidateInstall(extensionID string) bool {
	return b.checkGrantedCapabilities(CapValidateInstalls, &map[string]string{ParamExtensionID: extensionID})
}

func (b *BarbradyCredentials) CanMonetizeExtensions() bool {
	barbradyGranted := b.BarbradyCanMonetizeExtensions()
	fallbackGranted := b.fallbackCredentials.CanMonetizeExtensions()

	b.compare("CanMonetizeExtensions", barbradyGranted, fallbackGranted)

	return fallbackGranted
}

func (b *BarbradyCredentials) BarbradyCanMonetizeExtensions() bool {
	return b.checkGrantedCapabilities(CapMonetizeExtensions, &map[string]string{})
}

// CanPerformPrivacyRequests Will only ever be true for S2S Credentials
func (b *BarbradyCredentials) CanPerformPrivacyRequests() bool {
	return false
}

// Barbrady Credentials Specific Code

func (b *BarbradyCredentials) RequestCapabilities(capabilities []string, params map[string]string) {
	b.capabilities = capabilities

	b.parameters = params
	// Merge UserID or ClientID into params
	if b.parameters[ParamClientID] == "" && b.parameters[ParamUserID] == "" {
		if b.UserID() != nil {
			b.parameters[ParamUserID] = *b.UserID()
		} else {
			b.parameters[ParamClientID] = b.ClientID()
		}
	}

	var resp *barbradytwirp.GetCapabilitiesResponse
	var respErr error

	_ = b.statsd.Inc("Barbrady_GetCapabilities", 1, 1.0)
	err := hystrix.Do(barbradyCapabilityLookupCircuit, func() error {
		start := time.Now()
		resp, respErr = b.client.GetCapabilities(context.Background(), &barbradytwirp.GetCapabilitiesRequest{
			Capabilities: b.capabilities,
			Parameters:   b.parameters,
		})
		duration := time.Since(start)
		_ = b.statsd.TimingDuration("Barbrady_GetCapabilities_Duration", duration, 1.0)
		return respErr
	}, nil)

	if err != nil {
		_ = b.statsd.Inc("Barbrady_GetCapabilities_Error", 1, 1.0)
		logrus.WithFields(logrus.Fields{
			"capabilities": b.capabilities,
			"parameters":   b.parameters,
			"endpoint":     b.endpoint,
			"error":        err,
		}).Error("Error Getting Capabilities from Barbrady")

		b.shouldCompare = false

		return
	}

	for _, capability := range resp.Capabilities {
		b.grantedCapabilities[capability.Name] = capability
	}
}

func (b *BarbradyCredentials) String() string {
	return fmt.Sprintf("%+v", b.grantedCapabilities)
}

func (b *BarbradyCredentials) FallbackCredentials() Credentials {
	return b.fallbackCredentials
}

func (b *BarbradyCredentials) SetEndpoint(endpoint string) {
	b.endpoint = endpoint
}

func (b *BarbradyCredentials) compare(method string, barbradyIsGranted, cartmanIsGranted bool) {
	if !b.isSourceCartman() && !b.alreadyCompared[method] && b.shouldCompare {
		b.alreadyCompared[method] = true

		if barbradyIsGranted != cartmanIsGranted {
			_ = b.statsd.Inc(fmt.Sprintf("Barbrady_Compare_Mismatch_%s", method), 1, 1.0)
			logrus.WithFields(logrus.Fields{
				"authorization_method":   method,
				"barbrady_capabilities":  b.String(),
				"barbrady_is_granted":    barbradyIsGranted,
				"endpoint":               b.endpoint,
				"fallback_is_granted":    cartmanIsGranted,
				"fallback_capabilities":  b.fallbackCredentials.String(),
				"fallback_type":          reflect.TypeOf(b.fallbackCredentials),
				"parameters":             b.parameters,
				"requested_capabilities": b.capabilities,
			}).Warn("Barbrady Authorization did not match Fallback Credentials!")
		} else {
			_ = b.statsd.Inc(fmt.Sprintf("Barbrady_Compare_Match_%s", method), 1, 1.0)
		}
	}
}

func (b *BarbradyCredentials) checkGrantedCapabilities(capability string, requiredParams *map[string]string) bool {
	if cap := b.grantedCapabilities[capability]; cap != nil && cap.Granted {
		if requiredParams != nil {
			for key, value := range *requiredParams {
				if b.parameters[key] != value {
					return false
				}
			}
		}

		return true
	}
	return false
}

func (b *BarbradyCredentials) isWhitelistMember(whitelist wp.Action) bool {
	var isMember bool

	err := hystrix.Do(barbardyWhitelistLookupCircuit, func() error {
		isMember = b.whitelistAdapter.HasMember(whitelist, b)
		return nil
	}, nil)
	if err != nil {
		logrus.WithError(err).Error("Barbrady could not request isWhitelistMember")
	}

	return isMember
}

func (b *BarbradyCredentials) isWhitelistViewer(whitelist wp.Action) bool {
	var isViewer bool

	err := hystrix.Do(barbardyWhitelistLookupCircuit, func() error {
		isViewer = b.whitelistAdapter.IsViewer(whitelist, b)
		return nil
	}, nil)
	if err != nil {
		logrus.WithError(err).Error("Barbrady could not request isWhitelistViewer")
	}

	return isViewer
}

func (b *BarbradyCredentials) isWhitelistEditor(whitelist wp.Action) bool {
	var isEditor bool

	err := hystrix.Do(barbardyWhitelistLookupCircuit, func() error {
		isEditor = b.whitelistAdapter.IsEditor(whitelist, b)
		return nil
	}, nil)
	if err != nil {
		logrus.WithError(err).Error("Barbrady could not request isWhitelistEditor")
	}

	return isEditor
}

func (b *BarbradyCredentials) isSourceCartman() bool {
	return b.UserID() == nil && b.ClientID() == ""
}
