package auth

import (
	"fmt"

	ga "code.justin.tv/common/goauthorization"
	whitelistProtocol "code.justin.tv/gds/gds/extensions/whitelist/protocol"
)

const (
	// DEPRECATED - this should be a whitelist with cartman access control
	iPhoneClientID = "85lcqzxpb9bqu9z6ga1ol55du"
	iPadClientID   = "p9lhq6azjkdl72hs5xnt3amqu7vv8k2"
)

type cartmanCredentials struct {
	decoder ga.Decoder
	token   *ga.AuthorizationToken
}

var _ Credentials = &cartmanCredentials{}

// NewCartmanCredentials returns Credentials based on the incoming token
func NewCartmanCredentials(decoder ga.Decoder, token *ga.AuthorizationToken) Credentials {
	return &cartmanCredentials{decoder, token}
}

func (c *cartmanCredentials) String() string {
	return fmt.Sprintf("%+v", c.token.Claims)
}

func (c *cartmanCredentials) CanAssumeIdentity(userID string) bool {
	return userID == c.token.GetSubject() && userID != ""
}

func (c *cartmanCredentials) ClientID() string {
	return c.token.GetClientID()
}

func (c *cartmanCredentials) UserID() *string {
	if sub := c.token.GetSubject(); sub != "" {
		return &sub
	}
	return nil
}

func (c *cartmanCredentials) FlatUserID() string {
	return c.token.GetSubject()
}

func (c *cartmanCredentials) HomeChannel() string {
	// operating on behalf of a specific channel?
	if claim, err := c.token.GetTokenClaims(CapEditInstalls); err == nil {
		if id, found := claim[ParamChannelID]; found {
			if channel, ok := id.(string); ok {
				return channel
			}
		}
	}
	return c.token.GetSubject()
}

func (c *cartmanCredentials) CanInstallInto(channelID string) bool {
	return c.checkToken(ga.CapabilityClaims{CapEditInstalls: ga.CapabilityClaim{
		ParamChannelID: channelID,
	}})
}

func (c *cartmanCredentials) CanActivateOn(channelID string) bool {
	return c.checkToken(ga.CapabilityClaims{CapEditActivations: ga.CapabilityClaim{
		ParamChannelID: channelID,
	}})
}

func (c *cartmanCredentials) CanInstallAllExtensions() bool {
	return c.checkTokenForAction(whitelistProtocol.InstallAllExtensions)
}

func (c *cartmanCredentials) CanReviewExtensions() bool {
	return c.checkTokenForAction(whitelistProtocol.ReviewExtensions)
}

func (c *cartmanCredentials) CanModerateExtensions() bool {
	return c.checkTokenForAction(whitelistProtocol.ModerateExtensions)
}

func (c *cartmanCredentials) CanReviveExtensions() bool {
	return c.checkTokenForAction(whitelistProtocol.ReviveExtensions)
}

func (c *cartmanCredentials) CanCreateExtensions() bool {
	return c.checkToken(ga.CapabilityClaims{CapOnboardIntoExtensions: ga.CapabilityClaim{}}) || c.checkTokenForAction(whitelistProtocol.CreateExtensions)
}

func (c *cartmanCredentials) CanMonetizeExtensions() bool {
	return c.checkTokenForAction(whitelistProtocol.MonetizeExtensions)
}

func (c *cartmanCredentials) CanViewAllExtensions() bool {
	return c.checkTokenForAction(whitelistProtocol.ViewAllExtensions)
}

func (c *cartmanCredentials) CanHardDeleteExtensions() bool {
	return c.checkTokenForAction(whitelistProtocol.HardDeleteExtensions)
}

func (c *cartmanCredentials) CanEditWhitelist(action whitelistProtocol.Action) bool {
	return c.checkTokenForAction(whitelistProtocol.EditAllWhitelists) ||
		c.checkTokenForAccess(whitelistProtocol.EditWhitelistPrefix, action)
}

func (c *cartmanCredentials) CanListWhitelists() bool {
	return c.checkTokenForAction(whitelistProtocol.ViewAllWhitelists)
}

func (c *cartmanCredentials) CanViewWhitelist(action whitelistProtocol.Action) bool {
	return c.checkTokenForAction(whitelistProtocol.ViewAllWhitelists) ||
		c.checkTokenForAccess(whitelistProtocol.ViewWhitelistPrefix, action)
}

func (c *cartmanCredentials) CanCurateAllCategories() bool {
	return c.checkTokenForAction(whitelistProtocol.CurateAllCategories)
}

func (c *cartmanCredentials) CanEditAllCategories() bool {
	return c.checkTokenForAction(whitelistProtocol.EditAllCategories)
}

func (c *cartmanCredentials) CanCurateAllGames() bool {
	return c.checkTokenForAction(whitelistProtocol.CurateAllCategories)
}

func (c *cartmanCredentials) CanCreateVersion(extensionID string) bool {
	return c.checkToken(ga.CapabilityClaims{CapCreateVersion: ga.CapabilityClaim{
		ParamExtensionID: extensionID,
	}})
}

func (c *cartmanCredentials) CanReadDeveloperOnlyData(extensionID string) bool {
	return c.checkToken(ga.CapabilityClaims{CapCreateVersion: ga.CapabilityClaim{
		ParamExtensionID: extensionID,
	}})
}

func (c *cartmanCredentials) CanEditVersion(extensionID string) bool {
	return c.checkToken(ga.CapabilityClaims{CapEditVersion: ga.CapabilityClaim{
		ParamExtensionID: extensionID,
	}})
}

func (c *cartmanCredentials) CanValidateInstall(extensionID string) bool {
	return c.checkToken(ga.CapabilityClaims{CapValidateInstalls: ga.CapabilityClaim{
		ParamExtensionID: extensionID,
	}})
}

func (c *cartmanCredentials) CanSkipMobileCheck(extensionID string) bool {
	// DEPRECATED - convert to actual Cartman security check
	return c.ClientID() != iPhoneClientID && c.ClientID() != iPadClientID
}

func (c *cartmanCredentials) CanPerformPrivacyRequests() bool {
	return false
}

func (c *cartmanCredentials) checkToken(claims ga.CapabilityClaims) bool {
	return c.decoder.Validate(c.token, claims) == nil
}

func (c *cartmanCredentials) checkTokenForAction(action whitelistProtocol.Action) bool {
	return c.checkToken(ga.CapabilityClaims{string(action): ga.CapabilityClaim{}})
}

func (c *cartmanCredentials) checkTokenForAccess(prefix string, action whitelistProtocol.Action) bool {
	return c.checkToken(ga.CapabilityClaims{
		prefix: ga.CapabilityClaim{whitelistProtocol.CartmanParamAction: string(action)},
	})
}

func (c cartmanCredentials) CanSeeHiddenExtensions() bool {
	return c.CanModerateExtensions() || c.CanReviewExtensions() || c.CanEditAllCategories() || c.CanViewAllExtensions()
}

// Barbrady Only Method. Keeping here for now until we remove Cartman authorization.
func (c *cartmanCredentials) RequestCapabilities([]string, map[string]string) {}

