package auth

import (
	"fmt"

	"code.justin.tv/gds/gds/extensions/ems/utils"
	wp "code.justin.tv/gds/gds/extensions/whitelist/protocol"
)

var _ Credentials = (*FakeCredentials)(nil)

// FakeCredentials are used for tests and when no credentials are found.
type FakeCredentials struct {
	Role            ExtensionRole
	ClientIDValue   string
	UserIDValue     string
	AllowChannelID  string
	OwnedExtensions []string
	SkipMobileCheck bool
}

var _ Credentials = &FakeCredentials{}

// NoPermissions returns a FakeCredentials that gives no permissions.
func NoPermissions() Credentials {
	return &FakeCredentials{}
}

// AllPermissions returns a FakeCredentials that allows any extension action.
func AllPermissions() Credentials {
	return &FakeCredentials{
		Role:            Developer | Seer | Reviewer | Moderator | Creator | Cleric | Administrator | Categorizer | Authorizer | Installer | Monetizer | Deleter | PrivacyRequester,
		AllowChannelID:  "*",
		OwnedExtensions: []string{"*"},
		UserIDValue:     "insecure", // garbage user id because legacy search doesn't want an empty string
		SkipMobileCheck: true,
	}
}

// CanListWhitelists returns true if the credentials can view the list of available whitelists
func (f *FakeCredentials) CanListWhitelists() bool {
	return f.validateRole(Authorizer)
}

// CanViewWhitelist returns true if the credentials can view the membership of the whitelist
func (f *FakeCredentials) CanViewWhitelist(wp.Action) bool {
	return f.validateRole(Authorizer)
}

// CanEditWhitelist returns true if the credentials can edit the membership of the whitelist
func (f *FakeCredentials) CanEditWhitelist(wp.Action) bool {
	return f.validateRole(Administrator)
}

// CanViewAllExtensions returns true if the credentials can view any extension even in test
func (f *FakeCredentials) CanViewAllExtensions() bool {
	return f.validateRole(Seer) || f.CanReviewExtensions()
}

// CanInstallAllExtensions returns true if the credentials can install any visible extension even in test
func (f *FakeCredentials) CanInstallAllExtensions() bool {
	return f.validateRole(Installer) || f.CanReviewExtensions()
}

// CanAssumeIdentity returns true if the credentials can represent the user
func (f *FakeCredentials) CanAssumeIdentity(userID string) bool {
	return userID != "" && f.UserIDValue == userID
}

// CanInstallInto returns true if the credentials can modify the channel installs/activations
func (f *FakeCredentials) CanInstallInto(channelID string) bool {
	return channelID != "" && (f.AllowChannelID == "*" || f.AllowChannelID == channelID || f.UserIDValue == channelID)
}

// CanActivateOn returns true if the credentials can modify the channel activations
func (f *FakeCredentials) CanActivateOn(channelID string) bool {
	return channelID != "" && (f.AllowChannelID == "*" || f.AllowChannelID == channelID || f.UserIDValue == channelID)
}

// CanReviewExtensions returns true if the credentials can review extensions
func (f *FakeCredentials) CanReviewExtensions() bool {
	return f.validateRole(Reviewer)
}

// CanModerateExtensions returns true if the credentials can moderate extensions
func (f *FakeCredentials) CanModerateExtensions() bool {
	return f.validateRole(Moderator)
}

// CanReviveExtensions returns true if the credentials can move extensions from deleted or rejected
func (f *FakeCredentials) CanReviveExtensions() bool {
	return f.validateRole(Cleric)
}

// CanCreateExtensions returns true if the credentials can create extensions
func (f *FakeCredentials) CanCreateExtensions() bool {
	return f.validateRole(Creator)
}

// CanCurateAllCategories returns true if the credentials can curate which extensions are in a category
func (f *FakeCredentials) CanCurateAllCategories() bool {
	return f.validateRole(Categorizer)
}

// CanEditAllCategories returns true if the credentials can edit details about a category
func (f *FakeCredentials) CanEditAllCategories() bool {
	return f.validateRole(Categorizer)
}

// CanCurateAllGames returns true if the credentials can curate which extensions are in a game
func (f *FakeCredentials) CanCurateAllGames() bool {
	return f.validateRole(Categorizer)
}

// CanCreateVersion returns true if the credentials can create versions for the extension
func (f *FakeCredentials) CanCreateVersion(extensionID string) bool {
	return f.hasRole(Developer, extensionID) && f.validateRole(Creator)
}

// CanReadDeveloperOnlyData returns true if the credentials can read developer data (including PII)
func (f *FakeCredentials) CanReadDeveloperOnlyData(extensionID string) bool {
	return f.hasRole(Developer, extensionID)
}

// CanEditVersion returns true if the credentials can edit versions for the extension
func (f *FakeCredentials) CanEditVersion(extensionID string) bool {
	return f.hasRole(Reviewer|Developer, extensionID)
}

// CanValidateInstall returns true if the credentials can verify install prerequisites for the extension
func (f *FakeCredentials) CanValidateInstall(extensionID string) bool {
	return f.hasRole(Developer, extensionID)
}

// CanSkipMobileCheck returns true if the client can ignore the display on iOS check
func (f *FakeCredentials) CanSkipMobileCheck(extensionID string) bool {
	return f.SkipMobileCheck || (extensionID != "" && extensionID == f.ClientID())
}

// CanMonetizeExtensions returns true if the credentials can monetize extensions
func (f *FakeCredentials) CanMonetizeExtensions() bool {
	return f.validateRole(Monetizer)
}

// CanHardDeleteExtensions returns true if the credentials can hard delete extensions
func (f *FakeCredentials) CanHardDeleteExtensions() bool {
	return f.validateRole(Deleter)
}

// CanPerformPrivacyRequests returns true if the credentials can audit or hard delete a user and their extensions
func (f *FakeCredentials) CanPerformPrivacyRequests() bool {
	return f.validateRole(PrivacyRequester)
}

// ClientID returns the authenticated client for this action
func (f *FakeCredentials) ClientID() string {
	return f.ClientIDValue
}

// UserID returns the authenticated user for this action if one exists or nil of it was an anonymous or service request
func (f *FakeCredentials) UserID() *string {
	if f.UserIDValue == "" {
		return nil
	}
	return &f.UserIDValue
}

// FlatUserID returns the authenticated user for this action if one exists or "" of it was an anonymous or service request
func (f *FakeCredentials) FlatUserID() string {
	return f.UserIDValue
}

// HomeChannel returns the channel that should be used to check for prediction if the credentials could install
// the extension on the user's channel -- different from GetUserID() because this may be an editor context.
func (f *FakeCredentials) HomeChannel() string { return f.UserIDValue }

func (f *FakeCredentials) validateRole(role ExtensionRole) bool {
	return f.Role&role == role
}

func (f *FakeCredentials) hasRole(role ExtensionRole, extID string) bool {
	if (role&Developer == Developer) && utils.StringIn(extID, f.OwnedExtensions) || utils.StringIn("*", f.OwnedExtensions) {
		return true
	}
	role = role &^ Developer
	// developer is per-extension and checked above
	return f.Role&role != 0
}

func (f *FakeCredentials) String() string {
	owned := "["
	for _, v := range f.OwnedExtensions {
		if len(owned) > 1 {
			owned = owned + ","
		}
		owned = owned + v
	}
	owned = owned + "]"
	return fmt.Sprintf("FakeCredentials(%v,%v,%v,%v,%v,%v)", f.Role, f.ClientIDValue, f.UserIDValue, f.AllowChannelID, owned, f.SkipMobileCheck)
}

func (f FakeCredentials) CanSeeHiddenExtensions() bool {
	return f.CanModerateExtensions() || f.CanReviewExtensions() || f.CanEditAllCategories() || f.CanViewAllExtensions()
}

// RequestCapabilities - Barbrady Only Method
func (f *FakeCredentials) RequestCapabilities([]string, map[string]string) {}

