package auth

import (
	"fmt"

	ga "code.justin.tv/common/goauthorization"
	wp "code.justin.tv/gds/gds/extensions/whitelist/protocol"
)

type gqlCredentials struct {
	token   *ga.AuthorizationToken
	adapter *whitelistAdapter
}

var _ Credentials = &gqlCredentials{}

// NewGQLCredentials creates a new credential that is specially tailored for GQL
func NewGQLCredentials(token *ga.AuthorizationToken, adapter *whitelistAdapter) Credentials {
	return &gqlCredentials{token, adapter}
}

func (g *gqlCredentials) String() string {
	return fmt.Sprintf("%+v", g.token.Claims)
}

func (g *gqlCredentials) ClientID() string    { return g.token.GetClientID() }
func (g *gqlCredentials) HomeChannel() string { return g.token.GetSubject() }

func (g *gqlCredentials) UserID() *string {
	userID := g.token.GetSubject()
	if userID == "" {
		return nil
	}
	return &userID
}

func (g *gqlCredentials) FlatUserID() string {
	return g.token.GetSubject()
}

func (g *gqlCredentials) CanAssumeIdentity(userID string) bool {
	return userID != "" && userID == g.token.GetSubject()
}

// NOTE : this doesn't allow channel editors; cartman does
func (g *gqlCredentials) CanInstallInto(channelID string) bool {
	return g.token.IsFirstPartyClient() && channelID == g.token.GetSubject()
}

// NOTE : this doesn't allow channel editors; cartman does
func (g *gqlCredentials) CanActivateOn(channelID string) bool {
	return g.token.IsFirstPartyClient() && channelID == g.token.GetSubject()
}

// NOTE : this assumes all first party clients should have this power...
func (g *gqlCredentials) CanViewAllExtensions() bool {
	return g.checkWhitelistIfFirstParty(wp.ViewAllExtensions) || g.CanReviewExtensions()
}

// NOTE : this assumes all first party clients should have this power...
func (g *gqlCredentials) CanInstallAllExtensions() bool {
	return g.checkWhitelistIfFirstParty(wp.InstallAllExtensions) || g.CanReviewExtensions()
}

// NOTE : this assumes all first party clients should have this power...
func (g *gqlCredentials) CanReviewExtensions() bool {
	return g.checkWhitelistIfFirstParty(wp.ReviewExtensions)
}

// NOTE : this assumes all first party clients should have this power...
func (g *gqlCredentials) CanCreateExtensions() bool {
	return g.checkWhitelistIfFirstParty(wp.CreateExtensions)
}

// NOTE : this assumes all first party clients should have this power...
func (g *gqlCredentials) CanMonetizeExtensions() bool {
	return g.checkWhitelistIfFirstParty(wp.MonetizeExtensions)
}

// CanCurateAllCategories returns true if the credentials can curate which extensions are in a category
func (g *gqlCredentials) CanCurateAllCategories() bool {
	return g.checkWhitelistIfFirstParty(wp.CurateAllCategories)
}

func (g *gqlCredentials) CanEditAllCategories() bool {
	return g.checkWhitelistIfFirstParty(wp.EditAllCategories)
}

// CanCurateAllGames returns true if the credentials can curate which extensions are in a game
func (g *gqlCredentials) CanCurateAllGames() bool {
	return g.checkWhitelistIfFirstParty(wp.CurateAllCategories)
}

func (g *gqlCredentials) CanEditWhitelist(wp.Action) bool                  { return false }
func (g *gqlCredentials) CanListWhitelists() bool                          { return false }
func (g *gqlCredentials) CanViewWhitelist(wp.Action) bool                  { return false }
func (g *gqlCredentials) CanModerateExtensions() bool                      { return false }
func (g *gqlCredentials) CanReviveExtensions() bool                        { return false }
func (g *gqlCredentials) CanCreateVersion(extensionID string) bool         { return false }
func (g *gqlCredentials) CanReadDeveloperOnlyData(extensionID string) bool { return false }
func (g *gqlCredentials) CanEditVersion(extensionID string) bool           { return false }
func (g *gqlCredentials) CanValidateInstall(extensionID string) bool       { return false }
func (g *gqlCredentials) CanHardDeleteExtensions() bool                    { return false }
func (g *gqlCredentials) CanPerformPrivacyRequests() bool                  { return false }

// GQL doesn't have a mobile filter. Use a sane default here.
func (g *gqlCredentials) CanSkipMobileCheck(extensionID string) bool { return false }

func (g *gqlCredentials) checkWhitelistIfFirstParty(action wp.Action) bool {
	if g.token.IsFirstPartyClient() {
		return g.adapter.HasMember(action, g)
	}
	return false
}

func (g gqlCredentials) CanSeeHiddenExtensions() bool {
	return g.CanModerateExtensions() || g.CanReviewExtensions() || g.CanEditAllCategories() || g.CanViewAllExtensions()
}

// Barbrady Only Method
func (g *gqlCredentials) RequestCapabilities([]string, map[string]string) {}

