package auth

import (
	"math/rand"
	"net/http"
	"time"

	ga "code.justin.tv/common/goauthorization"
	"code.justin.tv/gds/gds/extensions"
	wdata "code.justin.tv/gds/gds/extensions/whitelist/data"
	wp "code.justin.tv/gds/gds/extensions/whitelist/protocol"
	"github.com/afex/hystrix-go/hystrix"
	statsdClient "github.com/cactus/go-statsd-client/statsd"

	barbradyTwirp "code.justin.tv/amzn/TwitchExtensionsBarbradyTwirp"
)

const (
	// CartmanAudience is the expected audience for EMS capabilities signed by Cartman
	CartmanAudience = "code.justin.tv/gds/ems"
	// GQLAudience is the expected audience for GQL tokens signed by Cartman`
	GQLAudience = "code.justin.tv/web/cartman"
	// WhitelistAudience is the expected audience for Whitelist capabilities signed by Cartman
	WhitelistAudience = wp.CartmanJwtAudience
	// CartmanIssuer is the expected issuer for capabilities signed by Cartman
	CartmanIssuer = "code.justin.tv/web/cartman"
	// Circuits for requests to Barbrady Service and Whitelist Service
	barbradyCapabilityLookupCircuit = "barbrady_capability_lookup"
	barbardyWhitelistLookupCircuit  = "barbrady_whitelist_lookup"
)

var (
	gqlDecoder, _       = ga.NewDecoder("HS512", GQLAudience, CartmanIssuer, []byte("unused"))
	whitelistDecoder, _ = ga.NewDecoder("HS512", WhitelistAudience, CartmanIssuer, []byte("unused"))
)

type jwtHandler struct {
	decoder                 ga.Decoder
	adapter                 *whitelistAdapter
	barbradyClient          barbradyTwirp.TwitchExtensionsBarbrady
	barbradySamplingPercent int32
	statsd                  statsdClient.Statter
}

// NewJWTHandler takes a decoder and whitelist and serves token based
// authorization. It uses the audience to determine if a token is coming from
// Cartman or GQL.
func NewJWTHandler(decoder ga.Decoder, manager wdata.WhitelistManager, barbradyClient barbradyTwirp.TwitchExtensionsBarbrady, barbradySamplingPercent int32, statsd statsdClient.Statter) Handler {
	hystrix.ConfigureCommand(barbradyCapabilityLookupCircuit, hystrix.CommandConfig{
		ErrorPercentThreshold: 60,
		MaxConcurrentRequests: 4000,
		SleepWindow:           1000,
		Timeout:               250,
	})

	hystrix.ConfigureCommand(barbardyWhitelistLookupCircuit, hystrix.CommandConfig{
		ErrorPercentThreshold: 60,
		MaxConcurrentRequests: 4000,
		SleepWindow:           1000,
		Timeout:               100,
	})

	return &jwtHandler{decoder, newWhitelistAdapter(manager), barbradyClient, barbradySamplingPercent, statsd}
}

func (j *jwtHandler) GetCredentials(r *http.Request) (Credentials, error) {
	token, err := j.decoder.ParseToken(r)
	if err == ga.ErrNoAuthorizationToken {
		return NoPermissions(), nil // anonymous request
	}

	// we couldn't parse it, so reject it
	if err != nil {
		return nil, extensions.ErrUnauthorized
	}

	// check validity as an EMS capability token
	err = j.decoder.Validate(token, ga.CapabilityClaims{})
	if err == nil {
		cartmanCreds := NewCartmanCredentials(j.decoder, token)

		if j.shouldUseBarbrady() {
			return NewBarbradyCredentials(cartmanCreds, j.barbradyClient, j.adapter, j.statsd), nil
		}
		return cartmanCreds, nil
	}

	// check validity as a Whitelist capability token
	if whitelistDecoder.Validate(token, ga.CapabilityClaims{}) == nil {
		cartmanCreds := NewCartmanCredentials(whitelistDecoder, token)

		if j.shouldUseBarbrady() {
			return NewBarbradyCredentials(cartmanCreds, j.barbradyClient, j.adapter, j.statsd), nil
		}
		return cartmanCreds, nil
	}

	// check validity as a GQL token
	if gqlDecoder.Validate(token, ga.CapabilityClaims{}) == nil {
		gqlCreds := NewGQLCredentials(token, j.adapter)

		if j.shouldUseBarbrady() {
			return NewBarbradyCredentials(gqlCreds, j.barbradyClient, j.adapter, j.statsd), nil
		}
		return gqlCreds, nil
	}

	// if it's not a valid token in any context, reject it
	return nil, extensions.ErrUnauthorized
}

func (j *jwtHandler) shouldUseBarbrady() bool {
	rand.Seed(time.Now().UnixNano())
	return int32(rand.Intn(100)) < j.barbradySamplingPercent
}
