package auth

import (
	"context"
	"net/http"

	wauth "code.justin.tv/gds/gds/extensions/whitelist/auth"
	"github.com/jixwanwang/apiutils"
)

var (
	credentialsMarker = 42                 // this value is never used, the address is important
	credentialsKey    = &credentialsMarker // unique within this process, private to this package
)

// StoreHandler builds a goji middleware that automatically places a
// Credentials into the context of each request or early outs with an
// unauthorized error.
func StoreHandler(handler Handler) func(http.Handler) http.Handler {
	return func(inner http.Handler) http.Handler {
		return http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
			creds, err := handler.GetCredentials(r)
			if err != nil {
				apiutils.ServeError(w, apiutils.NewErrorResponse(http.StatusUnauthorized, err.Error()))
			} else {
				ctx := Store(r.Context(), creds)
				inner.ServeHTTP(w, r.WithContext(ctx))
			}
		})
	}
}

// Store wraps the input context with one containing the specified credentials
func Store(ctx context.Context, creds Credentials) context.Context {
	base := wauth.Store(ctx, creds) // need to store at location wauth.Load expects, too
	return context.WithValue(base, credentialsKey, creds)
}

// Load retrieves credentials from a request context.
func Load(ctx context.Context) Credentials {
	if value, ok := ctx.Value(credentialsKey).(Credentials); ok {
		return value
	}
	return &FakeCredentials{}
}
