package auth

import (
	"context"
	"log"
	"time"

	"code.justin.tv/gds/gds/golibs/cache"

	payday "code.justin.tv/commerce/payday/client"
	"code.justin.tv/commerce/payday/models/api"
	wdata "code.justin.tv/gds/gds/extensions/whitelist/data"
	wp "code.justin.tv/gds/gds/extensions/whitelist/protocol"
	owl "code.justin.tv/web/owl/client"
	"code.justin.tv/web/owl/oauth2"
	"github.com/afex/hystrix-go/hystrix"
	statsdClient "github.com/cactus/go-statsd-client/statsd"
)

const readSubscriptionsScope = "channel:read:subscriptions"

// Permissions interface
type Permissions interface {
	CanLinkExternalContent(extensionID string) bool
	CanDisplayMobileView(creds Credentials, extensionID string) bool
	CanInstallAllExtensions(channelID string) bool
	CanMonetize(channelID string) bool
	CanCheckSubscriptionStatus(extensionClientID, channelID string) bool
}

type permissions struct {
	adapter     *whitelistAdapter
	paydayCache *cache.Local
	owlCache    *cache.Local
	statsd      statsdClient.Statter
}

var _ Permissions = &permissions{}

// NewPermissions makes a new Permissions with the provided manager
func NewPermissions(manager wdata.WhitelistManager, owlClient owl.Client, paydayClient payday.Client, cacheDuration time.Duration, statsd statsdClient.Statter) Permissions {
	hystrix.ConfigureCommand("ems_auth_lookup_channel_eligibility", hystrix.CommandConfig{
		Timeout:               900,
		SleepWindow:           1000,
		ErrorPercentThreshold: 60,
		MaxConcurrentRequests: 2000,
	})
	hystrix.ConfigureCommand("ems_auth_authorizations_for_broadcaster_id", hystrix.CommandConfig{
		Timeout:               900,
		SleepWindow:           1000,
		ErrorPercentThreshold: 60,
		MaxConcurrentRequests: 2000,
	})
	return &permissions{
		adapter:     newWhitelistAdapter(manager),
		paydayCache: cache.NewLocal(lookupChannelEligibility(paydayClient, statsd), cacheDuration),
		owlCache:    cache.NewLocal(authorizationsForBroadcasterID(owlClient, statsd), cacheDuration),
		statsd:      statsd,
	}
}

// CanLinkExternalContent checks if you can outbound link
func (f *permissions) CanLinkExternalContent(extensionID string) bool {
	return f.adapter.HasClient(wp.LinkExternalContent, extensionID)
}

// CanDisplayMobileView checks if the given extension is known to be okay to show on an iOS device
func (f *permissions) CanDisplayMobileView(creds Credentials, extensionID string) bool {
	return creds.CanSkipMobileCheck(extensionID) || f.adapter.HasClient(wp.DisplayOnIOS, extensionID)
}

// CanInstallAllExtensions checks if the given channel is allowed to install any extension
func (f *permissions) CanInstallAllExtensions(channelID string) bool {
	return f.adapter.HasUser(wp.InstallAllExtensions, channelID)
}

// CanMonetize checks if a the given channel has access to monetization (isPartner/isAffiliate)
func (f *permissions) CanMonetize(channelID string) bool {
	f.statsd.Inc("Payday_CacheLookup", 1, 1.0) // 100% sample rate
	cached, err := f.paydayCache.Get(channelID)
	if err != nil {
		return false
	}

	if paydayInfo, ok := cached.(*api.ChannelEligibleResponse); ok {
		return paydayInfo.Eligible
	}

	return false
}

// CanCheckSubscriptionStatus checks if the given oauthToken is still valid for the channelID/extensionClientID
func (f *permissions) CanCheckSubscriptionStatus(extensionClientID, broadcasterID string) bool {
	f.statsd.Inc("Owl_CacheLookup", 1, 1.0) // 100% sample rate
	cached, err := f.owlCache.Get(broadcasterID)
	if err != nil {
		return false
	}

	if authorizations, ok := cached.([]*oauth2.Authorization); ok {
		for _, authorization := range authorizations {
			if authorization.ClientIDCanonical == extensionClientID {
				for _, scope := range authorization.Scope {
					if scope == readSubscriptionsScope {
						return true
					}
				}
				// Can short-circuit here since there is only one record per client ID.
				return false
			}
		}
	}

	return false
}

func lookupChannelEligibility(client payday.Client, statsd statsdClient.Statter) cache.Source {
	return func(channelID string) (interface{}, error) {
		var resp *api.ChannelEligibleResponse
		var uErr error

		start := time.Now()
		err := hystrix.Do("ems_auth_lookup_channel_eligibility", func() error {
			resp, uErr = client.GetChannelInfo(context.Background(), channelID, nil)
			return uErr
		}, nil)
		totalTime := time.Since(start)

		statsd.TimingDuration("Payday_GetChannelInfo_Timing", totalTime, 1.0) // 100% sample rate
		statsd.Inc("Payday_CacheMiss", 1, 1.0)                                // 100% sample rate

		if err != nil {
			statsd.Inc("Payday_GetChannelInfo_Error", 1, 1.0)
			log.Printf("Encountered error with Payday: %v\n", err)
			return nil, err
		}

		return resp, err
	}
}

func authorizationsForBroadcasterID(client owl.Client, statsd statsdClient.Statter) cache.Source {
	return func(broadcasterID string) (interface{}, error) {
		var resp []*oauth2.Authorization
		var uErr error

		start := time.Now()
		err := hystrix.Do("ems_auth_authorizations_for_broadcaster_id", func() error {
			resp, uErr = client.Authorizations(context.Background(), broadcasterID, nil)
			return uErr
		}, nil)
		totalTime := time.Since(start)

		statsd.TimingDuration("Owl_Authorizations_Timing", totalTime, 1.0) // 100% sample rate
		statsd.Inc("Owl_CacheMiss", 1, 1.0)                                // 100% sample rate

		if err != nil {
			statsd.Inc("Owl_Authorizations_Error", 1, 1.0)
			log.Printf("Encountered error with Owl: %v\n", err)
			return nil, err
		}

		return resp, err
	}
}
