package auth

import (
	"fmt"
	"strings"

	"code.justin.tv/amzn/TwitchS2S2/s2s2"
	wp "code.justin.tv/gds/gds/extensions/whitelist/protocol"
)

const (
	ScopeViewAllWhitelists      = "view_all_whitelists"
	ScopePerformPrivacyRequests = "perform_privacy_requests"
)

type S2SCredentials struct {
	authorizedSubject s2s2.AuthorizedSubject
}

func NewS2SCredentials(authorizedSubject s2s2.AuthorizedSubject) Credentials {
	return &S2SCredentials{
		authorizedSubject: authorizedSubject,
	}
}

// CanListWhitelists permission not necessary
func (s *S2SCredentials) CanListWhitelists() bool {
	return s.canViewAllWhitelists()
}

// CanSeeHiddenExtensions permission not necessary
func (s *S2SCredentials) CanSeeHiddenExtensions() bool {
	return false
}

// CanViewWhitelist is necessary to view members on a whitelist
func (s *S2SCredentials) CanViewWhitelist(wp.Action) bool {
	return s.canPerformPrivacyRequests() || s.canViewAllWhitelists()
}

// CanEditWhitelist permission not necessary
func (s *S2SCredentials) CanEditWhitelist(wp.Action) bool {
	return false
}

// CanViewAllExtensions necessary to query all extensions
func (s *S2SCredentials) CanViewAllExtensions() bool {
	return s.canPerformPrivacyRequests()
}

// CanInstallAllExtensions permission not necessary
func (s *S2SCredentials) CanInstallAllExtensions() bool {
	return false
}

// CanAssumeIdentity permission not necessary
func (s *S2SCredentials) CanAssumeIdentity(userID string) bool {
	return false
}

// CanInstallInto not necessary
func (s *S2SCredentials) CanInstallInto(channelID string) bool {
	return false
}

// CanActivateOn returns true if the credentials can modify the channel activations
// CanActivateOn is needed to create the channel document to determine which extensions to uninstall and
//   remove from the channel.
func (s *S2SCredentials) CanActivateOn(channelID string) bool {
	return s.canPerformPrivacyRequests()
}

// CanReviewExtensions permission is not necessary
func (s *S2SCredentials) CanReviewExtensions() bool {
	return false
}

// CanModerateExtensions permission is not necessary
func (s *S2SCredentials) CanModerateExtensions() bool {
	return false
}

// CanReviveExtensions permission is not necessary
func (s *S2SCredentials) CanReviveExtensions() bool {
	return false
}

// CanCreateExtensions permission is not necessary
func (s *S2SCredentials) CanCreateExtensions() bool {
	return false
}

// CanCurateAllCategories permission not necessary
func (s *S2SCredentials) CanCurateAllCategories() bool {
	return false
}

// CanEditAllCategories permission not necessary
func (s *S2SCredentials) CanEditAllCategories() bool {
	return false
}

// CanCurateAllGames permission not necessary
func (s *S2SCredentials) CanCurateAllGames() bool {
	return false
}

// CanCreateVersion permission not necessary
func (s *S2SCredentials) CanCreateVersion(extensionID string) bool {
	return false
}

// CanReadDeveloperOnlyData permission not necessary
func (s *S2SCredentials) CanReadDeveloperOnlyData(extensionID string) bool {
	return false
}

// CanEditVersion permission not necessary
func (s *S2SCredentials) CanEditVersion(extensionID string) bool {
	return false
}

// CanValidateInstall permission not necessary
func (s *S2SCredentials) CanValidateInstall(extensionID string) bool {
	return false
}

// CanSkipMobileCheck permission not necessary
func (s *S2SCredentials) CanSkipMobileCheck(extensionID string) bool {
	return false
}

// CanMonetizeExtensions permission not necessary
func (s *S2SCredentials) CanMonetizeExtensions() bool {
	return false
}

// CanHardDeleteExtensions returns true if the credentials can hard delete extensions
// Need to be able to
func (s *S2SCredentials) CanHardDeleteExtensions() bool {
	return s.canPerformPrivacyRequests()
}

// CanPerformPrivacyRequests allows S2S callees to trigger the AuditUser & HardDeleteUser endpoints
func (s *S2SCredentials) CanPerformPrivacyRequests() bool {
	return s.canPerformPrivacyRequests()
}

// ClientID returns the authenticated client for this action
func (s *S2SCredentials) ClientID() string {
	return ""
}

// UserID returns the authenticated user for this action if one exists or nil of it was an anonymous or service request
func (s *S2SCredentials) UserID() *string {
	return nil
}

// FlatUserID returns a text representation of UserID, in this case empty string since UserID always returns nil
func (s *S2SCredentials) FlatUserID() string {
	return ""
}

// HomeChannel returns the channel that should be used to check for prediction if the credentials could install
// the extension on the user's channel -- different from GetUserID() because this may be an editor context.
func (s *S2SCredentials) HomeChannel() string { return "" }

// Barbrady Only Method
func (s *S2SCredentials) RequestCapabilities([]string, map[string]string) {}

// String returns the caller name
func (s *S2SCredentials) String() string {
	return s.authorizedSubject.TokenID()
}

func (s *S2SCredentials) canPerformPrivacyRequests() bool {
	return s.authorizedSubjectScopeContains(ScopePerformPrivacyRequests)
}

func (s *S2SCredentials) canViewAllWhitelists() bool {
	return s.authorizedSubjectScopeContains(ScopeViewAllWhitelists)
}

func (s *S2SCredentials) authorizedSubjectScopeContains(scope string) bool {
	return strings.Contains(fmt.Sprintf("%+v", s.authorizedSubject.Scope()), scope)
}
