package auth

import (
	"net/http"

	"code.justin.tv/amzn/TwitchS2S2/s2s2"
	"code.justin.tv/amzn/TwitchS2S2/s2s2/s2s2iface"
	"code.justin.tv/extensions/instrumentation"
)

type S2SProtector struct {
	s2s2Client s2s2iface.S2S2API
}

func NewS2SProtector(s2s2Client s2s2iface.S2S2API) *S2SProtector {
	return &S2SProtector{
		s2s2Client,
	}
}

// ProtectWithScope calls S2S client to first see if there is a valid request being made and then puts the
//   S2S credentials / scopes onto the context if it can find a caller.
func (s *S2SProtector) ProtectWithScope(scope string, handler http.Handler) http.Handler {
	setCreds := s.setCredentialsOnContextIfAuthorizationPresent(handler)

	if cast, ok := handler.(instrumentation.InstrumentedHandler); ok {
		return instrumentation.InstrumentHandler(
			cast.Name(),
			cast.Frequency(),
			s.s2s2Client.RequireScopes(
				setCreds,
				s.s2s2Client.CapabilityScope(scope),
			).PassthroughIfAuthorizationNotPresented(),
		)
	}
	return s.s2s2Client.RequireScopes(setCreds, s.s2s2Client.CapabilityScope(scope)).PassthroughIfAuthorizationNotPresented()
}

func (s *S2SProtector) setCredentialsOnContextIfAuthorizationPresent(handler http.Handler) http.Handler {
	return http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
		authorizedSubject, ok := s2s2.OptionalRequestSubject(r.Context())

		// Only use s2s creds if they were found on the request
		if ok {
			creds := NewS2SCredentials(authorizedSubject)
			ctx := Store(r.Context(), creds)
			handler.ServeHTTP(w, r.WithContext(ctx))
		} else {
			handler.ServeHTTP(w, r)
		}

	})
}
