package messages

import (
	"encoding/json"
	"time"

	"code.justin.tv/gds/gds/extensions/ems/protocol"
	"code.justin.tv/gds/gds/golibs/event"
)

// AssetMessage implements event.Message; Sequence will increase monotonically
// for a specific extension version and id; not all values will be present, and
// sequences numbers from different extension versions can't be reliably compared.
type AssetMessage struct {
	Format            int                    `json:"format_version"`
	Sequence          int                    `json:"sequence"`
	UserID            string                 `json:"user_id"`
	Time              time.Time              `json:"time"`
	Operation         string                 `json:"operation"`
	Tags              []string               `json:"tags"`
	AssetInfo         *AssetInfo             `json:"extension"`
	DeprecatedVersion string                 `json:"deprecated_version,omitempty"`
	Priors            map[string]interface{} `json:"priors"`
}

type AssetInfo struct {
	ExtensionID string
	AssetHash   string
	UploadedBy  string
	UploadedAt  *time.Time
	FileSize    string
	FileName    string
}

// // Create a new extension message
func NewAssetInfoMessage(ai *protocol.AssetInfo, seq int, change event.Change) *AssetMessage {

	return &AssetMessage{Operation: string(change), Sequence: seq, AssetInfo: protocolToMesageAssetInfo(ai), Time: time.Now().Truncate(time.Second)}
}

func protocolToMesageAssetInfo(ai *protocol.AssetInfo) *AssetInfo {
	if ai == nil {
		return nil
	}
	return &AssetInfo{
		ExtensionID: ai.ExtensionID,
		AssetHash:   ai.AssetHash,
		UploadedBy:  ai.UploadedBy,
		UploadedAt:  ai.UploadedAt,
		FileSize:    ai.FileSize,
		FileName:    ai.FileName,
	}
}

// // Deserialize an asset message
func LoadAssetInfoMessage(change event.Change, format int, body []byte) (event.Message, error) {
	ext := &AssetMessage{}
	if err := json.Unmarshal(body, ext); err != nil {
		return nil, err
	}
	ext.Operation = string(change)
	return ext, nil
}

func (e *AssetMessage) FormatVersion() int   { return 1 }
func (e *AssetMessage) Topic() event.Topic   { return AssetTopic }
func (e *AssetMessage) Change() event.Change { return event.Change(e.Operation) }
func (e *AssetMessage) Body() interface{}    { return e }
