package messages

import (
	"encoding/json"
	"time"

	"code.justin.tv/gds/gds/extensions/models"
	"code.justin.tv/gds/gds/golibs/event"
)

// ChannelMessage implements event.Message. Sequence will increment monotonically
// within each channel id
type ChannelMessage struct {
	Format           int            `json:"format_version"`
	Sequence         int            `json:"sequence"`
	UserID           string         `json:"user_id"`
	Time             time.Time      `json:"time"`
	Operation        string         `json:"operation"`
	Tags             []string       `json:"tags"`
	ChannelID        string         `json:"channel_id"`
	ExtensionID      string         `json:"extension_id"`
	ExtensionVersion *string        `json:"extension_version,omitempty"` // empty if a release install
	Details          InstallDetails `json:"install_details"`
	DynamicDetails   DynamicDetails `json:"dynamic_details"`
}

type InstallDetails struct {
	Configuration  string   `json:"approved_configuration"`
	ApprovedScopes []string `json:"approved_scopes"`
	AnchorType     string   `json:"anchor_type"`
	Slot           string   `json:"slot"`
	X              int      `json:"x"`
	Y              int      `json:"y"`
}

// Details used for dynamic management events
type DynamicDetails struct {
	IsCauseDynamic bool   `json:"dynamic_caused"`
	GameID         string `Json:"game_id"`
}

// NewChannelMessage creates a new channel message
func NewChannelMessage(channelID string, seq int, change event.Change) *ChannelMessage {
	return &ChannelMessage{
		Format:    1,
		Sequence:  seq,
		Operation: string(change),
		ChannelID: channelID,
		Time:      time.Now().Truncate(time.Second),
		DynamicDetails: DynamicDetails{
			IsCauseDynamic: IsCauseDynamic(change),
		},
	}
}

// LoadChannelMessage deserializes an existing channel message
func LoadChannelMessage(change event.Change, format int, body []byte) (event.Message, error) {
	ch := &ChannelMessage{}
	if err := json.Unmarshal(body, ch); err != nil {
		return nil, err
	}
	ch.Operation = string(change)
	return ch, nil
}

func (c *ChannelMessage) FormatVersion() int   { return c.Format }
func (c *ChannelMessage) Topic() event.Topic   { return ChannelTopic }
func (c *ChannelMessage) Change() event.Change { return event.Change(c.Operation) }
func (c *ChannelMessage) Body() interface{}    { return c }

// IsActive provides a helper for clients to determine if an installation is Active.
func (i *InstallDetails) IsActive(ext *models.Extension) bool {
	if ext.RequiredConfigurationString != "" && ext.RequiredConfigurationString != i.Configuration {
		return false
	}

	for _, scope := range ext.RequiredBroadcasterAbilities {
		if !i.HasScope(scope) {
			return false
		}
	}

	foundAnchor := false
	for _, anchor := range ext.SupportedAnchorTypes() {
		if anchor == i.AnchorType {
			foundAnchor = true
		}
	}

	return foundAnchor
}

// HasScope returns true if the provided scope is found in the list of broadcaster approved scopes
func (i *InstallDetails) HasScope(scope string) bool {
	for _, s := range i.ApprovedScopes {
		if s == scope {
			return true
		}
	}
	return false
}

// IsCauseDynamic indicates if a given change was automatic due to the dynamic management settings.
func IsCauseDynamic(c event.Change) bool {
	switch c {
	case OnDynamicActivated, OnDynamicDeactivated, OnDynamicMaybeActivated:
		return true
	default:
		return false
	}
}
