package messages

import (
	"encoding/json"
	"time"

	"code.justin.tv/gds/gds/extensions/models" // TODO : break this association
	"code.justin.tv/gds/gds/golibs/event"
)

// ExtensionMessage implements event.Message; Sequence will increase monotonically
// for a specific extension version and id; not all values will be present, and
// sequences numbers from different extension versions can't be reliably compared.
type ExtensionMessage struct {
	Format            int                    `json:"format_version"`
	Sequence          int                    `json:"sequence"`
	UserID            string                 `json:"user_id"`
	Time              time.Time              `json:"time"`
	Operation         string                 `json:"operation"`
	Tags              []string               `json:"tags"`
	Extension         *models.Extension      `json:"extension"`
	DeprecatedVersion string                 `json:"deprecated_version,omitempty"`
	Priors            map[string]interface{} `json:"priors"`
}

// Create a new extension message
func NewExtensionMessage(ext *models.Extension, seq int, change event.Change) *ExtensionMessage {
	return &ExtensionMessage{Operation: string(change), Sequence: seq, Extension: ext, Time: time.Now().Truncate(time.Second)}
}

// Deserialize an extension message
func LoadExtensionMessage(change event.Change, format int, body []byte) (event.Message, error) {
	ext := &ExtensionMessage{}
	if err := json.Unmarshal(body, ext); err != nil {
		return nil, err
	}
	ext.Operation = string(change)
	return ext, nil
}

func (e *ExtensionMessage) FormatVersion() int   { return 1 }
func (e *ExtensionMessage) Topic() event.Topic   { return ExtensionTopic }
func (e *ExtensionMessage) Change() event.Change { return event.Change(e.Operation) }
func (e *ExtensionMessage) Body() interface{}    { return e }
