package data

import "code.justin.tv/gds/gds/extensions"

// AnchorType defines the behavior of a specific extension display style
type AnchorType string

const (
	// Hidden anchors are used for testing purposes only and are invisible to users
	Hidden AnchorType = "hidden"
	// VideoOverlay anchors are placed directly on top of streaming video
	VideoOverlay AnchorType = "video_overlay"
	// Panel anchors appear in the channel page below the video
	Panel AnchorType = "panel"
	// Component anchors appear on top of streaming video
	Component AnchorType = "component"
)

// ToAnchorType converts a string to an AnchorType
func ToAnchorType(name string) AnchorType {
	switch name {
	case string(VideoOverlay):
		return VideoOverlay
	case string(Panel):
		return Panel
	case string(Component):
		return Component
	default:
		return Hidden
	}
}

// MaxSlots returns the maximum slots that an AnchorType will support per channel
// TODO : consider making this data driven
func (a AnchorType) MaxSlots() int {
	switch a {
	case Panel:
		return extensions.MaxPanelExtensions
	case Component:
		return extensions.MaxComponentExtensions
	default:
		return 1
	}
}

// SupportsSlotCount returns true if the AnchorType supports the given slot count
func (a AnchorType) SupportsSlotCount(count int) error {
	if count > a.MaxSlots() {
		return extensions.NewMaxSlotsExceededError(string(a), a.MaxSlots())
	}
	return nil
}

// InSlice returns true if the anchor type is in the supplied slice
func (a AnchorType) InSlice(slice []AnchorType) bool {
	for _, e := range slice {
		if a == e {
			return true
		}
	}
	return false
}
