package data

import (
	"code.justin.tv/gds/gds/extensions"
	"code.justin.tv/gds/gds/extensions/ems/auth"
	"code.justin.tv/gds/gds/extensions/models"
)

// ExtensionState is an type that represents parts of the extension lifecycle
type ExtensionState string

const (
	// InTest extensions are currently being worked on by a developer.
	InTest = ExtensionState("test")
	// InReview extensions are being looked at by Twitch.
	InReview = ExtensionState("review")
	// Rejected extensions are permanently rejected; no action by the
	// developer will make them acceptable.
	Rejected = ExtensionState("rejected")
	// Approved extensions are ready to be released, at the developer's
	// convenience.
	Approved = ExtensionState("approved")
	// Released extensions are currently available to users.  Only one
	// version of an extension can be in this state at any time.
	Released = ExtensionState("released")
	// Deprecated extensions were once released, but a newer version now exists.
	Deprecated = ExtensionState("deprecated")
	// PendingAction extensions are ones that were reviewed by Twitch and
	// returned to the developer for fixing.
	PendingAction = ExtensionState("pending_action")
	// AssetsUploaded extensions have had their assets uploaded already to S3.
	// This state mirrors the deprecated ReadyForReview state, but doesn't incur
	// an implicit upload
	AssetsUploaded = ExtensionState("assets_uploaded")
	// Deleted extensions have been deleted; this is a soft delete so we can
	// undelete things.
	Deleted = ExtensionState("deleted")
)

var (
	// HumanNameByState maps ExtensionState to human-readable text
	humanNameByState = map[ExtensionState]string{
		InTest:         "Testing",
		InReview:       "In Review",
		Released:       "Released",
		Rejected:       "Rejected",
		Approved:       "Approved",
		Deprecated:     "Deprecated",
		PendingAction:  "Pending Action",
		AssetsUploaded: "Assets Uploaded",
		Deleted:        "Deleted",
	}
)

// AllExtensionStates returns the list of all extension states for tests to use in loops for testing various operations
// against extensions in various states.
func AllExtensionStates() []ExtensionState {
	var allStates = make([]ExtensionState, len(humanNameByState))
	i := 0
	for k, _ := range humanNameByState {
		allStates[i] = k
		i++
	}

	return allStates
}

// IsViewable returns true if creds grant viewing privilege
func (e ExtensionState) IsViewable(extensionID string, creds auth.Credentials) bool {
	switch e {
	case Deleted:
		return (auth.Cleric | auth.Developer | auth.Deleter).IsMember(extensionID, creds)
	case Rejected:
		return auth.Cleric.IsMember(extensionID, creds)
	case Released:
		return true
	}
	return (auth.Developer | auth.Seer).IsMember(extensionID, creds)
}

// AllowTesting returns true if the state allows the tester whitelist to be active
func (e ExtensionState) AllowTesting() bool {
	return e != Deprecated && e != Rejected
}

// IsInstallable returns true if the state allows installs
func (e ExtensionState) IsInstallable() bool {
	return e != Deleted && e != Deprecated && e != Rejected
}

// UseTestURI returns true if the state should use the test URI instead of the release one
func (e ExtensionState) UseTestURI() bool {
	return e == InTest || e == Deleted
}

// CanUpdateData returns true if the data can be modified
func (e ExtensionState) CanUpdateData() bool { return e == InTest }

// HasTransitionTo returns true if the dest state is reachable from this one
func (e ExtensionState) HasTransitionTo(dest ExtensionState) bool {
	_, ok := e.getTransitionTo(dest)
	return ok
}

// CanTransitionTo returns true if the specified creds allow the dest state to be reached
func (e ExtensionState) CanTransitionTo(dest ExtensionState, extensionID string, creds auth.Credentials) error {
	role, ok := e.getTransitionTo(dest)
	if !ok {
		return extensions.ErrInvalidStateTransition
	}
	if role.IsMember(extensionID, creds) {
		return nil
	}
	return extensions.ErrForbiddenStateTransition
}

// ToHumanName outputs a human readable version of the state
func (e ExtensionState) ToHumanName() string {
	return humanNameByState[e]
}

// FromName maps a string to an extension state
func FromName(name string) (ExtensionState, error) {
	state := ExtensionState(name)
	if _, ok := stateMap[state]; ok {
		return state, nil
	}
	return InTest, extensions.ErrUnknownState
}

// FromLegacy converts from a models.ExtensionState
func FromLegacy(in models.ExtensionState) ExtensionState {
	for out, compare := range legacyMap {
		if compare == in {
			return out
		}
	}
	return InTest
}

// FromLegacyWithFound is just like FromLegacy but gives you an indication that the state you were looking for wasn't
// found instead of returning a default of InTest.
func FromLegacyWithFound(in models.ExtensionState) (*ExtensionState, bool) {
	for out, compare := range legacyMap {
		if compare == in {
			return &out, true
		}
	}
	return nil, false
}

// ToLegacy converts to a models.ExtensionState
func (e ExtensionState) ToLegacy() models.ExtensionState {
	if out, ok := legacyMap[e]; ok {
		return out
	}
	return models.InTest
}

func (e ExtensionState) getTransitionTo(dest ExtensionState) (auth.ExtensionRole, bool) {
	legal, ok := stateMap[e]
	if !ok {
		return auth.NoRole, false
	}
	role, ok := legal[dest]
	if !ok {
		return auth.NoRole, false
	}
	return role, ok
}

// IsUnderReview is true for review states
func (e ExtensionState) IsUnderReview() bool {
	return e == InReview || e == Approved
}

// RoleMap lists roles who can transition to a state
type RoleMap map[ExtensionState]auth.ExtensionRole

var legacyMap = map[ExtensionState]models.ExtensionState{
	InTest:         models.InTest,
	InReview:       models.InReview,
	Rejected:       models.Rejected,
	Approved:       models.Approved,
	Released:       models.Released,
	Deprecated:     models.Deprecated,
	PendingAction:  models.PendingAction,
	AssetsUploaded: models.AssetsUploaded,
	Deleted:        models.Deleted,
}

// stateMap is, for each state, a list of states that an extension can transition to from that state, and the creds
// required to execute that transition.
var stateMap = map[ExtensionState]RoleMap{
	InTest: RoleMap{
		Deleted:        auth.Developer,
		InReview:       auth.Developer,
		AssetsUploaded: auth.Developer,
		PendingAction:  auth.Reviewer,
	},
	InReview: RoleMap{
		Deleted:       auth.Developer,
		InTest:        auth.Developer,
		Approved:      auth.Reviewer,
		PendingAction: auth.Reviewer,
	},
	Rejected: RoleMap{
		PendingAction: auth.Cleric,
	},
	Approved: RoleMap{
		Deleted:       auth.Developer,
		InTest:        auth.Developer,
		Released:      auth.Developer,
		PendingAction: auth.Reviewer,
	},
	Released: RoleMap{
		Deleted:       auth.Developer,
		PendingAction: auth.Moderator,
	},
	Deprecated: RoleMap{
		Deleted:  auth.Developer,
		Released: auth.Developer, // a.k.a. rollback
	},
	PendingAction: RoleMap{
		Deleted:  auth.Developer,
		InTest:   auth.Developer,
		Rejected: auth.Moderator,
	},
	AssetsUploaded: RoleMap{
		Deleted:  auth.Developer,
		InTest:   auth.Developer,
		InReview: auth.Developer,
	},
	Deleted: RoleMap{
		InTest: auth.Cleric,
	},
}
