package data

import (
	"strings"

	"code.justin.tv/gds/gds/extensions/ems/protocol"
)

// QueuedRequest is an extension of the Request interface which allows multiple extensions
// to be fetched. A queued request should specify the MissingBehavior upon construction
// and should add extensions through use of the Add() method.
type QueuedRequest interface {
	Request
	Add(id string, version *string)
}

type node struct {
	id      string
	version *string
	next    *node
}

type queuedRequest struct {
	mb MissingBehavior

	head    *node
	tail    *node
	current *node
}

func NewQueuedRequest(mb MissingBehavior) QueuedRequest {
	init := &node{}
	return &queuedRequest{
		mb:      mb,
		head:    init,
		tail:    init,
		current: init,
	}
}

func (r *queuedRequest) Add(id string, version *string) {
	r.tail.next = &node{id, version, nil}
	r.tail = r.tail.next
}

func (r *queuedRequest) Next() (string, *string, bool) {
	if r.current.next == nil {
		return "", nil, false
	}

	r.current = r.current.next
	return r.current.id, r.current.version, true
}

func (r *queuedRequest) OnMissing() MissingBehavior {
	return r.mb
}

func (r *queuedRequest) Reset() { r.current = r.head }

func ExtensionKeysToQueuedRequest(keysEncoded string) (QueuedRequest, error) {
	if keysEncoded == "" {
		return nil, protocol.ErrInvalidExtensionKey
	}

	qr := NewQueuedRequest(MissingBehaviorReturnNil)
	for _, encodedKey := range strings.Split(keysEncoded, ",") {
		key, err := protocol.ParseExtensionKey(encodedKey)
		if err != nil {
			return nil, protocol.ErrInvalidExtensionKey
		}

		qr.Add(key.ID(), key.Version())
	}

	return qr, nil
}
