package data

// MissingBehavior represents the behavior a request should have when an extension is not found
type MissingBehavior string

const (
	// MissingBehaviorError means the request should fail when the extension is missing
	MissingBehaviorError MissingBehavior = "error"

	// MissingBehaviorReturnNil means the request return nil in place of the missing extension
	MissingBehaviorReturnNil MissingBehavior = "return_nil"

	// MissingBehaviorDrop means the request should act as if the extension wasn't even requested
	MissingBehaviorDrop MissingBehavior = "drop"
)

// Request represents an iterable struct used to fetch extensions.
type Request interface {
	// Reset returns the Request implementation back to it's original initialization state
	Reset()

	// Next iterates the request returning the `id`, `version`, and `ok`.
	// `id` will be a string which represents the clientID of an extensions. It is legal for an ID to
	// be an empty string. If `id` is an empty string while `ok` is `true`, the extensions should be
	// treated as "missing" by the caller with the behavior specified in OnMissing().
	// `version` will be a pointer to a string. If nil, the released version should be used.
	// `ok` is a boolean which is true if the current iterator has valid data
	Next() (string, *string, bool)

	// OnMissing returns the expected behavior for the entire request when fetching the
	// specified extension results in error.
	OnMissing() MissingBehavior
}

func ReleasedRequest(id string) Request           { return &simpleRequest{id, nil, false} }
func VersionedRequest(id, version string) Request { return &simpleRequest{id, &version, false} }
