package data

import (
	"sort"

	"code.justin.tv/gds/gds/extensions/ems/documents"
	"code.justin.tv/gds/gds/extensions/models"
)

// Sorts a set of active extensions by a predefined order for consistency in clients requesting the data
// The order of active extensions should be as follows:
// - Overlays
// - Components (1, 2)
// - Panels (1, 2, 3)
func SortActiveExtensions(doc *documents.InstalledExtensionsDocument) *documents.InstalledExtensionsDocument {
	out := documents.InstalledExtensionsDocument{
		InstalledExtensions: []*documents.InstalledExtensionDocument{},
	}

	extensionsByType := make(map[string][]*documents.InstalledExtensionDocument)

	for _, installation := range doc.InstalledExtensions {
		anchorType := ""
		if installation.InstallationStatus != nil &&
			installation.InstallationStatus.ActivationState == models.ActivationStateActive &&
			installation.InstallationStatus.Anchor != nil {
			anchorType = string(installation.InstallationStatus.Anchor.Value.Type())
		}

		extensionsByType[anchorType] = append(extensionsByType[anchorType], installation)
	}

	anchorTypeOrder := []string{string(VideoOverlay), string(Component), string(Panel)}

	for _, anchorTypeOrder := range anchorTypeOrder {
		anchoredExtensions := extensionsByType[anchorTypeOrder]
		sort.Slice(anchoredExtensions, func(i, j int) bool {
			extensionA := anchoredExtensions[i]
			extensionB := anchoredExtensions[j]
			if extensionA == nil || extensionB == nil {
				return false
			}

			// Currently relying on the string ordering of slot names for sorting within an anchor type
			// If we expand beyond allowing 10 extensions per anchor type, we'll want to revisit this.
			return extensionA.InstallationStatus.Anchor.Value.Slot() < extensionB.InstallationStatus.Anchor.Value.Slot()
		})

		out.InstalledExtensions = append(out.InstalledExtensions, anchoredExtensions...)
	}

	// Leave inactive extensions in their original sort order at the end of the document
	out.InstalledExtensions = append(out.InstalledExtensions, extensionsByType[""]...)

	return &out
}
