package protocol

import (
	"net/http"

	"code.justin.tv/gds/gds/golibs/errors"
)

const (
	// ErrActionNotFoundCode is used by ErrActionNotFound
	ErrActionNotFoundCode = "action_not_found"
	// ErrConcurrencyCode is used by ErrConcurrency
	ErrConcurrencyCode = "concurrency_violation"
	// ErrInternalServiceErrorCode is used by ErrInvalidCacheKey
	ErrInternalServiceErrorCode = "internal_service_error"
	// ErrInvalidMemberCode is used by ErrInvalidMember
	ErrInvalidMemberCode = "invalid_member_id"
	// ErrInvalidPagingTokenCode is used by ErrInvalidPagingToken
	ErrInvalidPagingTokenCode = "invalid_paging_token"
	// ErrMissingParameterCode is used by ErrMissingParameter
	ErrMissingParameterCode = "missing_parameter"
	// ErrNotAvailableCode is used by ErrNotAvailable
	ErrNotAvailableCode = "service_unavailable"
	// ErrNotImplementedCode is used by ErrNotImplemented
	ErrNotImplementedCode = "not_implemented"
	// ErrTooManyRequestsCode is used by ErrTooManyRequests
	ErrTooManyRequestsCode = "too_many_requests"
	// ErrUnauthorizedCode is used by ErrUnauthorized
	ErrUnauthorizedCode = "unauthorized"
)

var (
	// ErrConcurrency is returned if the whitelist store had a concurrency failure
	ErrConcurrency = errors.NewBuilder("Write collision; please try again").
			WithErrorCode(ErrConcurrencyCode).
			WithHTTPStatus(http.StatusConflict).
			Build()
	// ErrInvalidCacheKey is returned if an internal failure causes the cache to fail
	ErrInvalidCacheKey = errors.NewBuilder("Invalid cache key").
				WithErrorCode(ErrInternalServiceErrorCode).
				WithHTTPStatus(http.StatusInternalServerError).
				Build()
	// ErrInvalidMember is returned if an API call is made with an invalid member string
	ErrInvalidMember = errors.NewBuilder("Invalid member id").
				WithErrorCode(ErrInvalidMemberCode).
				WithHTTPStatus(http.StatusUnprocessableEntity).
				Build()
	// ErrInvalidPagingToken is returned if the whitelist store can't accept the specified paging token
	ErrInvalidPagingToken = errors.NewBuilder("Invalid paging token").
				WithErrorCode(ErrInvalidPagingTokenCode).
				WithHTTPStatus(http.StatusInternalServerError).
				Build()
	// ErrNotAvailable is returned if the whitelist service is currently unable to serve requests
	ErrNotAvailable = errors.NewBuilder("Whitelists are not available").
			WithErrorCode(ErrNotAvailableCode).
			WithHTTPStatus(http.StatusServiceUnavailable).
			Build()
	// ErrNotImplemented is returned if an API call is made against an incomplete server implementation
	ErrNotImplemented = errors.NewBuilder("Not implemented").
				WithErrorCode(ErrNotImplementedCode).
				WithHTTPStatus(http.StatusNotImplemented).
				Build()
	// ErrTooManyRequests is returned if the whitelist store can't handle the current request volume
	ErrTooManyRequests = errors.NewBuilder("Too many requests").
				WithErrorCode(ErrTooManyRequestsCode).
				WithHTTPStatus(http.StatusTooManyRequests).
				Build()
	// ErrUnauthorized is returned if an API call is made without proper credentials
	ErrUnauthorized = errors.NewBuilder("Unauthorized").
			WithErrorCode(ErrUnauthorizedCode).
			WithHTTPStatus(http.StatusUnauthorized).
			Build()
)

// ErrActionNotFound is returned if an API call is made with an invalid action
func ErrActionNotFound(action string) error { return &errActionNotFound{action} }

type errActionNotFound struct{ action string }

var _ errors.DetailsError = (*errActionNotFound)(nil)
var _ errors.HTTPStatusError = (*errActionNotFound)(nil)
var _ errors.ErrorCodeError = (*errActionNotFound)(nil)

func (e *errActionNotFound) Error() string   { return "Action \"" + e.action + "\"not found" }
func (*errActionNotFound) HTTPStatus() int   { return http.StatusNotFound }
func (*errActionNotFound) ErrorCode() string { return ErrActionNotFoundCode }
func (e *errActionNotFound) Details() errors.Details {
	return map[string]interface{}{"action": e.action}
}

// ErrMissingParameter is returned if an API call is missing a required parameter
func ErrMissingParameter(parameter string) error { return &errMissingParameter{parameter} }

type errMissingParameter struct{ parameter string }

var _ errors.DetailsError = (*errMissingParameter)(nil)
var _ errors.HTTPStatusError = (*errMissingParameter)(nil)
var _ errors.ErrorCodeError = (*errMissingParameter)(nil)

func (e *errMissingParameter) Error() string {
	return "Missing required parameter \"" + e.parameter + "\""
}
func (*errMissingParameter) HTTPStatus() int   { return http.StatusUnprocessableEntity }
func (*errMissingParameter) ErrorCode() string { return "missing_required_parameter" }
func (e *errMissingParameter) Details() errors.Details {
	return map[string]interface{}{"parameter": e.parameter}
}
