package protocol

import (
	"encoding/json"
	"sort"
	"strings"
)

const (
	clientPrefix = "c:"
	userPrefix   = "u:"
)

// Member defines a whitelist member -- it is a struct instead of
// a string alias to prevent silent conversion
type Member struct {
	value string
}

// Members defines a wrapper around member arrays
type Members []Member

var _ sort.Interface = Members{}

// Anonymous represents the unauthenticated user
func Anonymous() Member { return Member{} }

// UserMember converts a userID into a whitelist member; returns Anonymous if it
// is passed an empty string
func UserMember(userID string) Member {
	if userID == "" {
		return Anonymous()
	}
	return Member{userPrefix + userID}
}

// ClientMember converts a clientID into a whitelist member; returns Anonymous
// if it is passed an empty string
func ClientMember(clientID string) Member {
	if clientID == "" {
		return Anonymous()
	}
	return Member{clientPrefix + clientID}
}

// Illegal is a special value used to force marshal errors for unit testing
func Illegal() Member { return Member{"!"} }

// UserID extracts the user id from a whitelist member if present
func (m Member) UserID() (string, bool) {
	if strings.HasPrefix(m.value, userPrefix) {
		return m.value[2:], true
	}
	return "", false
}

// ClientID extracts the client id from a whitelist member if present
func (m Member) ClientID() (string, bool) {
	if strings.HasPrefix(m.value, clientPrefix) {
		return m.value[2:], true
	}
	return "", false
}

// Validate sanity checks whether a member is valid, optionally accepting the anonymous user
func (m Member) Validate(allowAnonymous bool) bool {
	if m.value == Anonymous().value {
		return allowAnonymous
	}
	return len(m.value) > 2 && (strings.HasPrefix(m.value, userPrefix) || strings.HasPrefix(m.value, clientPrefix))
}

// Less compares two Members
func (m Member) Less(m2 Member) bool {
	return m.value < m2.value
}

// Equal compares two values for equality
func (m Member) Equal(m2 Member) bool {
	return m.value == m2.value
}

// UnmarshalJSON allows Member to be transmitted as a string
func (m *Member) UnmarshalJSON(b []byte) error {
	var s string
	if err := json.Unmarshal(b, &s); err != nil {
		return err
	}
	if s == Illegal().value {
		return ErrInvalidMember
	}
	m.value = s
	return nil
}

// MarshalJSON allows Member to be transmitted as a string
func (m Member) MarshalJSON() ([]byte, error) {
	if m.Equal(Illegal()) {
		return nil, ErrInvalidMember
	}
	return json.Marshal(m.value)
}

// Contains allows searching for a Member in an array
func (m Members) Contains(v Member) bool {
	for _, v2 := range m {
		if v == v2 {
			return true
		}
	}
	return false
}

// GetID extracts the user or client id from the member
func (m Member) GetID() string {
	if id, ok := m.ClientID(); ok {
		return id
	}
	if id, ok := m.UserID(); ok {
		return id
	}

	return ""
}

// Len implements sort.Interface by returning the length of the slice
func (m Members) Len() int { return len([]Member(m)) }

// Less reports whether the element with
// index i should sort before the element with index j.
func (m Members) Less(i, j int) bool { return m[i].value < m[j].value }

// Swap swaps the elements with indexes i and j.
func (m Members) Swap(i, j int) { m[i], m[j] = m[j], m[i] }
