package cache

import (
	"time"

	"github.com/patrickmn/go-cache"
)

// Local cache collects parallel requests behind implicit promise that it will continue
// to serve until the internal cache expires its entry.  The internal cache runs
// a periodic sweep to clean out dead entries every 30 seconds. Calls to Get
// will block until the underlying promise has been fulfilled. Entries are keyed
// by an arbtirary string; one Local cache can serve many types of object as
// long as the key content is properly namespaced.
type Local struct {
	internal *cache.Cache
	source   AsyncSource
}

// NewLocal returns a cache given a source and refreshAfter duration
func NewLocal(source Source, refreshAfter time.Duration) *Local {
	return &Local{cache.New(refreshAfter, 30*time.Second), background(source)}
}

// NewAsync returns a cache given an AsyncSource and a refreshAfter duration
func NewAsync(source AsyncSource, refreshAfter time.Duration) *Local {
	return &Local{cache.New(refreshAfter, 30*time.Second), source}
}

// Reset completely flushes the cache
func (l *Local) Reset() {
	l.internal.Flush()
}

// Get will create or retrieve a promise for a value based on a string key. The
func (l *Local) Get(key string) (interface{}, error) {
	return l.Load(key).Get()
}

// Insert manually places a value the cache using the standard duration.
func (l *Local) Insert(key string, raw interface{}) {
	l.internal.Set(key, NewPromise().Set(raw, nil), cache.DefaultExpiration)
}

// Remove drops a cached promise, forcing a source query on the next request
func (l *Local) Remove(key string) {
	l.internal.Delete(key)
}

// Load fetches or creates an entry for the given key asynchronously
func (l *Local) Load(key string) Promise {
	e, found := l.internal.Get(key)
	if found {
		return e.(Promise)
	}
	value := l.source(key)
	l.internal.Set(key, value, cache.DefaultExpiration)
	return value
}

// convert sync to async
func background(source Source) AsyncSource {
	return func(key string) Promise {
		value := NewPromise()
		go func() {
			data, err := source(key)
			value.Set(data, err)
		}()
		return value
	}
}
