## Config

This package contains an abstract configuration system that provides typed values from a prioritized list of arbitrary origins wrapped in a common interface. Each `Source` can be reused in different `Config` objects according to which data a system is interested in reading (e.g. `Config.WithOverrides(<src>)` can append a source for an individual call without affecting the base configuration).

`Config.Snapshot()` can be called to get point-in-time configuration for use cases that should not change.  This will cause the underlying structure to be flattened into a single `Source` record.

For convenience, the actual settings inside a config can be retrieved in three ways:
* `c.Require<type>(key)` will panic if key is absent, useful during initial setup
* `c.Get<type>(key, _default)` will fall back to `_default` if the key is absent
* `c.TryGet<type>(key)` returns the same (value, found) pair as a golang `map` so you can handle absence with custom logic 

### Middleware

This package provides goji middleware that can inject configuration into each request; it also supports the ability to create per-request overrides. Call `StoreConfig(config)` to create the middleware and call `server.Use()` to place it into your server's middleware stack. The resulting configuration can be read using the `Load(context)` function. If you want to manually store a config instead of using the middleware, you can call `c.Store(context)` to generate an updated context; you will then need to manually insert the context into the request for the next handler in the stack.

### RefreshController

Many of the configuration sources allow polling to pick up changes. The `RefreshController` simplifies the management of configuration polling through a simple `Schedule(src)`/`Unschedule(src)` function pair that takes both default refresh timing and the ability to specify a timing in configuration--allowing a source to potentially dictate its own behavior. Clean shutdown is available via `UnscheduleAll()` which should be used with `defer`.

### Builders

#### NewConfig
Builds a `Config` given a comma delimited list of `Source`s that can be assembled from the other builders below.

#### NewRefreshController
Builds an empty `RefreshController` that is ready to schedule update polling.

#### FromRawValues
Builds a static `Source` from a golang `map[string]interface{}`. Supports nested maps (which are converted to dotted namespaces), integers except for uint64, floats, booleans, and strings.

#### FromJSON
Builds a static `Source` from a JSON string with the same type restrictions as `FromRawValues`

#### FromEC2
Builds a refreshable `Source` from EC2 instance tags; the value of `{config.ec2.prefix}` is used to filter tag keys and is removed from each key that is returned. Key names have ':' converted to '.' when they are saved.  If the prefix field is not present or empty, the system defaults to `twitch:ds:`.

#### FromS3
Builds a refreshable `Source` from an S3 bucket. There must be exactly one bucket in the account tagged with the magic value `twitch:ds:s3:type`==`configuration`. Configuration objects must be in `JSON` format and end with the suffix `.config.json` to be read. The object must also start with a prefix that can be set in one of two ways: either from the value of  `{config.s3.prefx}` or by constructing a path from the values `{app.env}/{app.app}/{app.comp}/config` if the prefix value is empty or missing.  In the event that the prefix and the constructed path are both absent, the builder will fail.

#### TestFromFilesystem
Builds a refreshable `Source` from a local json file. The path is immutable from the time of declaration, and this source type should not be deployed to production--when assembling configuration, the `app.env` value should be read from trusted sources to determine whether to include this type.
