package config

import (
	"context"

	"code.justin.tv/gds/gds/golibs/config/sources"
)

// Config is the public interface for the config package; an ordered list of
// arbitrary configuration sources can be combined into a single Config that
// can be queried for values or listened to for changes.
type Config interface {
	sources.Settings
	// GetBool returns a bool value if found or the supplied default
	GetBool(key string, _default bool) bool
	// GetFloat returns a float64 value if found or the supplied default
	GetFloat(key string, _default float64) float64
	// GetInt returns an int64 value if found or the supplied default
	GetInt(key string, _default int64) int64
	// GetString returns a string value if found or the supplied default
	GetString(key, _default string) string
	// GetSource reports the highest priority configuration source that contains
	// the key or null if the key is not present
	GetSource(key string) *string
	// RequireBool returns a bool value if found or panics
	RequireBool(key string) bool
	// RequireFloat returns a float64 value if found or panics
	RequireFloat(key string) float64
	// RequireInt returns an int64 value if found or panics
	RequireInt(key string) int64
	// RequireString returns a string value if found or panics
	RequireString(key string) string
	// Snapshot returns a point-in-time representation of a Config as a Source; the
	// provided number is appended to the snapshot name, allowing the caller to
	// distinguish between different values.
	Snapshot(number int) (Source, error)
	// List returns a list of source/key/value triples sorted by key for diagnostic
	// purposes
	List() Settings
	// Checksum returns an MD5 summary of config content that can be used for
	// change detection
	Checksum() Checksum
	// ToJSON dumps a flattened listing of all current values *without* redaction
	ToJSON() ([]byte, error)
	// WithOverrides creates an independent config that overlays the new source on
	// top of the previously reported values.
	WithOverrides(source sources.Source) Config
	// WithPrefix returns an independent config that injects the requested prefix
	// into requests, creating a subtree view of the underlying content.
	WithPrefix(prefix string) Config
	// GetRoot sheds any subtree prefix that has been applied and returns the full
	// configuration tree.
	GetRoot() Config
	// Store injects the Config into a copy of the input Context; use
	// Load(ctx) to retrieve a stored configuration.
	Store(ctx context.Context) context.Context
}
