package config

import (
	"context"
	"encoding/json"
	"fmt"

	"code.justin.tv/gds/gds/golibs/config/sources"
)

const sourceNotFound = "<not found>"

type config struct {
	sources []sources.Source
}

func (c *config) WithOverrides(source sources.Source) Config {
	size := len(c.sources) + 1
	sources := make([]sources.Source, size, size)
	sources[0] = source
	copy(sources[1:], c.sources)
	return NewConfig(sources...)
}

func (c *config) GetRoot() Config { return c }

func (c *config) WithPrefix(prefix string) Config {
	return &prefixedConfig{prefix, c}
}

func (c *config) TryGetBool(name string) (bool, bool) {
	for _, src := range c.sources {
		if val, ok := src.GetBool(name); ok {
			return val, true
		}
	}
	return false, false
}

func (c *config) GetBool(name string, _default bool) bool {
	if val, ok := c.TryGetBool(name); ok {
		return val
	}
	return _default
}

func (c *config) RequireBool(name string) bool {
	if val, ok := c.TryGetBool(name); ok {
		return val
	}
	panic(fmt.Sprintf("%v not found", name))
}

func (c *config) TryGetFloat(name string) (float64, bool) {
	for _, src := range c.sources {
		if val, ok := src.GetFloat(name); ok {
			return val, true
		}
	}
	return 0, false
}

func (c *config) GetFloat(name string, _default float64) float64 {
	if val, ok := c.TryGetFloat(name); ok {
		return val
	}
	return _default
}

func (c *config) RequireFloat(name string) float64 {
	if val, ok := c.TryGetFloat(name); ok {
		return val
	}
	panic(fmt.Sprintf("%v not found", name))
}

func (c *config) TryGetInt(name string) (int64, bool) {
	for _, src := range c.sources {
		if val, ok := src.GetInt(name); ok {
			return val, true
		}
	}
	return 0, false
}

func (c *config) GetInt(name string, _default int64) int64 {
	if val, ok := c.TryGetInt(name); ok {
		return val
	}
	return _default
}

func (c *config) RequireInt(name string) int64 {
	if val, ok := c.TryGetInt(name); ok {
		return val
	}
	panic(fmt.Sprintf("%v not found", name))
}

func (c *config) TryGetString(name string) (string, bool) {
	for _, src := range c.sources {
		if val, ok := src.GetString(name); ok {
			return val, true
		}
	}
	return "", false
}

func (c *config) GetString(name, _default string) string {
	if val, ok := c.TryGetString(name); ok {
		return val
	}
	return _default
}

func (c *config) RequireString(name string) string {
	if val, ok := c.TryGetString(name); ok {
		return val
	}
	panic(fmt.Sprintf("%v not found", name))
}

func (c *config) GetSource(name string) *string {
	for _, src := range c.sources {
		if _, ok := src.GetString(name); ok {
			name = src.GetName()
			return &name
		}
	}
	return nil
}

func (c *config) Snapshot(number int) (Source, error) {
	return buildSnapshot("", c.sources, number)
}

func (c *config) ToJSON() ([]byte, error) {
	return json.Marshal(merge("", c.sources))
}

func (c *config) List() Settings {
	return buildSettings("", c)
}

func (c *config) Checksum() Checksum {
	return buildChecksum("", c.sources)
}

func (c *config) Store(ctx context.Context) context.Context {
	return context.WithValue(ctx, contextKey, c)
}
