package config

import (
	"context"
	"encoding/json"

	"code.justin.tv/gds/gds/golibs/config/sources"
)

type prefixedConfig struct {
	prefix string
	inner  *config
}

func (p *prefixedConfig) fixup(name string) string {
	return p.prefix + "." + name
}

func (p *prefixedConfig) GetRoot() Config { return p.inner }

func (p *prefixedConfig) WithOverrides(source sources.Source) Config {
	size := len(p.inner.sources) + 1
	sources := make([]sources.Source, size, size)
	sources[0] = source
	copy(sources[1:], p.inner.sources)
	return &prefixedConfig{p.prefix, &config{sources}}
}

func (p *prefixedConfig) WithPrefix(prefix string) Config {
	return &prefixedConfig{p.fixup(prefix), p.inner}
}

func (p *prefixedConfig) TryGetBool(name string) (bool, bool) {
	return p.inner.TryGetBool(p.fixup(name))
}

func (p *prefixedConfig) GetBool(name string, _default bool) bool {
	return p.inner.GetBool(p.fixup(name), _default)
}

func (p *prefixedConfig) RequireBool(name string) bool {
	return p.inner.RequireBool(p.fixup(name))
}

func (p *prefixedConfig) TryGetFloat(name string) (float64, bool) {
	return p.inner.TryGetFloat(p.fixup(name))
}

func (p *prefixedConfig) GetFloat(name string, _default float64) float64 {
	return p.inner.GetFloat(p.fixup(name), _default)
}

func (p *prefixedConfig) RequireFloat(name string) float64 {
	return p.inner.RequireFloat(p.fixup(name))
}

func (p *prefixedConfig) TryGetInt(name string) (int64, bool) {
	return p.inner.TryGetInt(p.fixup(name))
}

func (p *prefixedConfig) GetInt(name string, _default int64) int64 {
	return p.inner.GetInt(p.fixup(name), _default)
}

func (p *prefixedConfig) RequireInt(name string) int64 {
	return p.inner.RequireInt(p.fixup(name))
}

func (p *prefixedConfig) TryGetString(name string) (string, bool) {
	return p.inner.TryGetString(p.fixup(name))
}

func (p *prefixedConfig) GetString(name, _default string) string {
	return p.inner.GetString(p.fixup(name), _default)
}

func (p *prefixedConfig) RequireString(name string) string {
	return p.inner.RequireString(p.fixup(name))
}

func (p *prefixedConfig) GetSource(name string) *string {
	return p.inner.GetSource(p.fixup(name))
}

func (p *prefixedConfig) Checksum() Checksum {
	return buildChecksum(p.fixup(""), p.inner.sources)
}

func (p *prefixedConfig) Snapshot(number int) (Source, error) {
	return buildSnapshot(p.fixup(""), p.inner.sources, number)
}

func (p *prefixedConfig) ToJSON() ([]byte, error) {
	return json.Marshal(merge(p.fixup(""), p.inner.sources))
}

func (p *prefixedConfig) List() Settings {
	return buildSettings(p.fixup(""), p.inner)
}

func (p *prefixedConfig) Store(ctx context.Context) context.Context {
	return context.WithValue(ctx, contextKey, p)
}
