package config

import (
	"sync"
	"time"
)

// RefreshErrorCallback is given asynchronous messages when a refresh attempt
// causes an error.
type RefreshErrorCallback func(RefreshableSource, error)

// RefreshSuccessCallback is given asynchronous messages when a refresh attempt
// has succeeded, regardless of whether values were updated.
type RefreshSuccessCallback func(Config)

// RefreshController simplifies management of configuration polling by providing
// a single object responsible for all scheduling.
type RefreshController interface {
	// Schedule creates or replaces a regularly scheduled refresh of the source
	Schedule(src RefreshableSource, cfg Config, tag string, fallback time.Duration)
	// Unschedule stops all scheduled refreshes
	Unschedule(src RefreshableSource)
	// UnscheduleAll stops all scheduled refreshes
	UnscheduleAll()
	// Listen attaches a callback to receive asynchronous notifications of refresh failures
	Listen(callback RefreshErrorCallback)
	// OnRefreshed attaches a callback to receive asynchronous notifications of refresh successes
	OnRefreshed(callback RefreshSuccessCallback)
}

type refreshController struct {
	entries   []*refreshEntry
	onError   []RefreshErrorCallback
	onSuccess []RefreshSuccessCallback
	mutex     sync.Mutex
}

func (r *refreshController) Schedule(src RefreshableSource, cfg Config, tag string, fallback time.Duration) {
	r.mutex.Lock()
	defer r.mutex.Unlock()
	entry := newRefreshEntry(src, cfg, tag, fallback, func() {
		for _, callback := range r.onSuccess {
			callback(cfg)
		}
	}, func(err error) {
		for _, callback := range r.onError {
			callback(src, err)
		}
	})
	r.remove(src)
	r.entries = append(r.entries, entry)
	entry.Start()
}

func (r *refreshController) Unschedule(src RefreshableSource) {
	r.mutex.Lock()
	defer r.mutex.Unlock()
	r.remove(src)
}

func (r *refreshController) UnscheduleAll() {
	r.mutex.Lock()
	defer r.mutex.Unlock()
	for _, entry := range r.entries {
		entry.Stop()
	}
	r.entries = nil
}

func (r *refreshController) OnRefreshed(callback RefreshSuccessCallback) {
	r.mutex.Lock()
	defer r.mutex.Unlock()
	r.onSuccess = append(r.onSuccess, callback)
}

func (r *refreshController) Listen(callback RefreshErrorCallback) {
	r.mutex.Lock()
	defer r.mutex.Unlock()
	r.onError = append(r.onError, callback)
}

func (r *refreshController) remove(src RefreshableSource) {
	filtered := make([]*refreshEntry, 0, len(r.entries))
	for _, entry := range r.entries {
		if entry.src != src {
			filtered = append(filtered, entry)
		} else {
			entry.Stop()
		}
	}
	r.entries = filtered
}
