package config

import (
	"sync"
	"time"
)

type refreshEntry struct {
	src      RefreshableSource
	cfg      Config
	tag      string
	fallback time.Duration
	refFunc  func()
	errFunc  func(error)
	stopped  bool
	timer    *time.Timer
	mutex    sync.Mutex
}

func newRefreshEntry(src RefreshableSource, cfg Config, tag string, fallback time.Duration, refFunc func(), errFunc func(error)) *refreshEntry {
	return &refreshEntry{
		src:      src,
		cfg:      cfg,
		tag:      tag,
		fallback: fallback,
		refFunc:  refFunc,
		errFunc:  errFunc,
		stopped:  true,
	}
}

func (r *refreshEntry) IsRunning() bool {
	r.mutex.Lock()
	defer r.mutex.Unlock()
	return r.timer != nil
}

// Start begins the refresh loop using the current settings, potentially
// canceling a scheduled refresh in progress
func (r *refreshEntry) Start() {
	r.mutex.Lock()
	defer r.mutex.Unlock()
	r.stopped = false
	if r.timer != nil {
		r.timer.Stop()
		r.timer = nil
	}
	r.schedule()
}

// Stop ends any current refresh loop
func (r *refreshEntry) Stop() {
	r.mutex.Lock()
	defer r.mutex.Unlock()
	r.stopped = true
	if r.timer != nil {
		r.timer.Stop()
		r.timer = nil
	}
}

func (r *refreshEntry) execute() {
	r.mutex.Lock()
	defer r.mutex.Unlock()
	if r.stopped {
		return
	}
	err := r.src.Refresh(r.cfg)
	if err != nil && r.errFunc != nil {
		go func() { r.errFunc(err) }()
	}
	if err == nil && r.refFunc != nil {
		go func() { r.refFunc() }()
	}
	// always schedule a retry even on error
	r.schedule()
}

func (r *refreshEntry) schedule() {
	r.timer = time.AfterFunc(r.duration(), r.execute)
}

func (r *refreshEntry) duration() time.Duration {
	duration := r.fallback
	if durStr, ok := r.cfg.TryGetString(r.tag); ok {
		// use default duration on duration string parse error
		override, err := time.ParseDuration(durStr)
		if err != nil && r.errFunc != nil {
			r.errFunc(err)
		} else {
			duration = override
		}
	}
	return duration
}
