package config

import (
	"fmt"
	"strings"
)

const redactText = "[redacted]"

// Setting represents a single value stored in a Config combined with the config
// Source that specified it.
type Setting struct {
	Source string
	Key    string
	Value  interface{}
}

// Redacted allows safe print of passwords or secrets when dumping configuration
func (s Setting) Redacted() interface{} {
	return redact(s.Key, s.Value)
}

// Settings is a []Setting with convenience overrides for how it is displayed.
// Default print creates a listing of values, PrintTable() adds some delimeters
// so the value jumps out in logs. Both automaticatlly redact any values with
// "password" or "secret" in the path.
type Settings struct {
	Values []Setting
	Prefix string
}

func newSettings(values ...Setting) Settings {
	if values == nil {
		values = []Setting{}
	}
	return Settings{values, ""}
}

func newPrefixedSettings(prefix string, values ...Setting) Settings {
	if values == nil {
		values = []Setting{}
	}
	return Settings{values, prefix}
}

func (s Settings) ShouldRedactAll() bool {
	return shouldRedact(s.Prefix)
}

// String overrides the default string behavior with an aligned print of one
// value per row delimited by newlines.
func (s Settings) String() string {
	sLen, keyLen, vLen := s.PrintLengths()
	return s.Pretty(sLen, keyLen, vLen, "", " ", "")
}

// BySource collects settings into a map of source to key-value mapping. This
// structure adapts well to JSON string marshaling in http routes. See the
// Display(ctx) function in middleware.go
func (s Settings) BySource() map[string]map[string]interface{} {
	bySource := make(map[string]map[string]interface{})
	for _, s := range s.Values {
		items, found := bySource[s.Source]
		if !found {
			items = make(map[string]interface{})
			bySource[s.Source] = items
		}
		items[s.Key] = s.Redacted()
	}
	return bySource
}

// PrintTable generates a series of values with vertical bar separators for
// ease of viewing in log dumps.
func (s Settings) PrintTable() string {
	sLen, keyLen, vLen := s.PrintLengths()
	vertical := strings.Repeat("-", sLen+keyLen+vLen+10)
	return fmt.Sprintf("%s\n%s\n%s", vertical, s.Pretty(sLen, keyLen, vLen, "| ", " | ", " |"), vertical)
}

// PrintLengths returns the length of each column for printing this collection,
// can could be used to build custom printout displays.
func (s Settings) PrintLengths() (int, int, int) {
	sLen, keyLen, vLen := 0, 0, 0
	redactAll := s.ShouldRedactAll()
	if redactAll {
		vLen = len(redactText)
	}
	for _, v := range s.Values {
		l1 := len(v.Source)
		if l1 > sLen {
			sLen = l1
		}
		l2 := len(v.Key)
		if l2 > keyLen {
			keyLen = l2
		}
		if !redactAll {
			l3 := len(fmt.Sprintf("%v", v.Value))
			if l3 > vLen {
				vLen = l3
			}
		}
	}
	return sLen, keyLen, vLen
}

// Pretty takes column widths and row markup and converts it into a block of
// newline delimited values
func (s Settings) Pretty(sLen, keyLen, vLen int, pre, sep, post string) string {
	var format string
	if post != "" {
		format = fmt.Sprintf("%s%%-%ds%s%%-%ds%s%%-%dv%s", pre, sLen, sep, keyLen, sep, vLen, post)
	} else {
		format = fmt.Sprintf("%s%%-%ds%s%%-%ds%s%%v", pre, sLen, sep, keyLen, sep)
	}
	lines := make([]string, 0, len(s.Values))
	redactAll := s.ShouldRedactAll()
	for _, v := range s.Values {
		if redactAll {
			lines = append(lines, fmt.Sprintf(format, v.Source, v.Key, redactText))
		} else {
			lines = append(lines, fmt.Sprintf(format, v.Source, v.Key, v.Redacted()))
		}
	}
	return strings.Join(lines, "\n")
}

func shouldRedact(key string) bool {
	return strings.Contains(key, "secret") || strings.Contains(key, "password")
}

func redact(key string, value interface{}) interface{} {
	if shouldRedact(key) {
		return redactText
	}
	return value
}
