package config

import (
	"fmt"
	"sort"
	"strings"

	"code.justin.tv/gds/gds/golibs/config/sources"
)

const emptySnapshotName = "empty"

func buildSnapshot(prefix string, src []sources.Source, number int) (Source, error) {
	name := buildSnapshotName(src, number)
	all := merge(prefix, src)
	return sources.NewStaticSource(name, all)
}

func buildSettings(prefix string, c *config) Settings {
	entries := merge(prefix, c.sources)
	s := make([]Setting, 0, len(entries))
	for k, v := range entries {
		s = append(s, listSetting(c, prefix, k, v))
	}
	sort.Slice(s, func(i, j int) bool { return s[i].Key < s[j].Key })
	return newPrefixedSettings(prefix, s...)
}

func listSetting(c *config, prefix, k string, v interface{}) Setting {
	s := c.GetSource(prefix + k)
	if s == nil {
		// possible due to race condition between merge and query
		return Setting{sourceNotFound, k, v}
	}
	return Setting{*s, k, v}
}

func merge(prefix string, sources []sources.Source) map[string]interface{} {
	// front of the slice overwrites end of the slice
	all := make(map[string]interface{})
	for i := len(sources) - 1; i >= 0; i-- {
		for k, v := range sources[i].GetAll() {
			if prefix == "" {
				all[k] = v
			} else if strings.HasPrefix(k, prefix) {
				all[k[len(prefix):]] = v
			}
		}
	}
	return all
}

func buildSnapshotName(sources []sources.Source, number int) string {
	if len(sources) < 1 {
		return emptySnapshotName
	}
	name := sources[0].GetName()
	for i := 1; i < len(sources); i++ {
		name += ":"
		name += sources[i].GetName()
	}
	return fmt.Sprintf("%v#%v", name, number)
}
