package discovery

import (
	"fmt"

	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/service/elasticache"
	"github.com/aws/aws-sdk-go/service/elasticache/elasticacheiface"
)

// ElastiCacheEndpointsByTag returns the list of EC Endpoints with the given tag
// also need to pass in the account ID and region, because AWS doesn't report
// back the ARN for the elasticache clusters, so we have to build them manually.
func ElastiCacheEndpointsByTag(svc elasticacheiface.ElastiCacheAPI, accountID, region, key, value string) ([]string, error) {
	marker := ""
	clusters := []*elasticache.CacheCluster{}

	for {
		input := elasticache.DescribeCacheClustersInput{
			ShowCacheNodeInfo: aws.Bool(true),
		}
		if marker != "" {
			input.Marker = aws.String(marker)
		}
		resp, err := svc.DescribeCacheClusters(&input)
		if err != nil {
			return nil, err
		}
		for _, cluster := range resp.CacheClusters {
			clusters = append(clusters, cluster)
		}
		if resp.Marker == nil {
			break
		}
		marker = *resp.Marker
	}

	endpoints := []string{}

	for _, cluster := range clusters {
		arn := buildARN(accountID, region, *cluster.CacheClusterId)
		tagResp, err := svc.ListTagsForResource(
			&elasticache.ListTagsForResourceInput{
				ResourceName: aws.String(arn),
			},
		)
		if err != nil {
			return nil, err
		}
		for _, tag := range tagResp.TagList {
			if *tag.Key == key && *tag.Value == value {
				for _, node := range cluster.CacheNodes {
					endpoints = append(endpoints, *node.Endpoint.Address)
				}
			}
		}
	}
	return endpoints, nil
}

func buildARN(accountID, region, cluster string) string {
	return fmt.Sprintf("arn:aws:elasticache:%s:%s:cluster:%s", region, accountID, cluster)
}
