package discovery

import (
	"github.com/aws/aws-sdk-go/service/elasticsearchservice"
	"github.com/aws/aws-sdk-go/service/elasticsearchservice/elasticsearchserviceiface"
)

type domainData struct {
	status *elasticsearchservice.ElasticsearchDomainStatus
	tags   *elasticsearchservice.ListTagsOutput
}

// ESDomains maps ARNs to attributes for query
type ESDomains map[string]domainData

func Domains(svc elasticsearchserviceiface.ElasticsearchServiceAPI) (ESDomains, error) {
	resp, err := svc.ListDomainNames(&elasticsearchservice.ListDomainNamesInput{})
	if err != nil {
		return nil, err
	}
	names := make([]*string, len(resp.DomainNames))
	for i, domain := range resp.DomainNames {
		names[i] = domain.DomainName
	}
	descriptionsResp, err := svc.DescribeElasticsearchDomains(&elasticsearchservice.DescribeElasticsearchDomainsInput{
		DomainNames: names,
	})
	if err != nil {
		return nil, err
	}
	out := make(ESDomains)
	for _, description := range descriptionsResp.DomainStatusList {
		if description.ARN == nil {
			continue
		}
		tagResp, err := svc.ListTags(&elasticsearchservice.ListTagsInput{
			ARN: description.ARN,
		})
		if err != nil {
			return nil, err
		}
		out[*description.ARN] = domainData{
			description,
			tagResp,
		}
	}
	return out, nil
}

func (e ESDomains) EndpointsForTag(key, value string) []string {
	out := []string{}
	for _, data := range e {
		for _, tag := range data.tags.TagList {
			if tag.Key != nil && tag.Value != nil && *tag.Key == key && *tag.Value == value {
				if data.status.Endpoint != nil {
					out = append(out, *data.status.Endpoint)
				}
			}
		}
	}
	return out
}
