package event

type Coordinator struct {
	requireListener bool
	listeners       map[Topic][]Listener
}

// NewCoordinator creates an event coordinator; if requireListener is true, the
// coordinator will only mark sources handled if a listener was able to read the
// messages from the source successfully
func NewCoordinator(requireListener bool) *Coordinator {
	return &Coordinator{requireListener, make(map[Topic][]Listener)}
}

func (c *Coordinator) Register(l Listener) error {
	for _, t := range l.GetTopics() {
		c.listeners[t] = append(c.listeners[t], l)
	}
	return nil
}

func (c *Coordinator) GetTopics() []Topic {
	topics := make([]Topic, 0, len(c.listeners))
	for k, _ := range c.listeners {
		topics = append(topics, k)
	}
	return topics
}

func (c *Coordinator) OnUpdate(src Source) error {
	for _, topic := range src.UnhandledTopics() {
		listeners, ok := c.listeners[topic]
		if ok {
			msgs := src.UnhandledMessages(topic)
			for _, l := range listeners {
				if err := l.OnMessages(msgs); err != nil {
					return err
				}
			}
		}
		if ok || !c.requireListener {
			if err := src.MarkHandled(topic); err != nil {
				return err
			}
		}
	}
	return nil
}
