package event

import (
	"encoding/base64"
	"encoding/json"
	"errors"
	"fmt"
	"strconv"
	"strings"
)

var ErrMalformedMessage = errors.New("Message is missing required information")

func Serialize(msgs ...Message) ([]string, error) {
	out := make([]string, 0, len(msgs))
	for _, msg := range msgs {
		body, err := json.Marshal(msg.Body())
		if err != nil {
			return nil, err
		}
		out = append(out, fmt.Sprintf("%v|%v|%v|%v", msg.Topic(), msg.Change(), msg.FormatVersion(), base64.RawURLEncoding.EncodeToString(body)))
	}
	return out, nil
}

func Deserialize(factory *MessageFactory, data ...string) ([]Message, error) {
	if len(data) < 1 {
		return nil, nil
	}
	out := make([]Message, 0, len(data))
	for _, value := range data {
		parts := strings.Split(value, "|")
		if len(parts) < 4 {
			return nil, ErrMalformedMessage
		}
		fmt, err := strconv.ParseInt(parts[2], 10, 32)
		if err != nil {
			return nil, ErrMalformedMessage
		}
		body, err := base64.RawURLEncoding.DecodeString(parts[3])
		if err != nil {
			return nil, ErrMalformedMessage
		}
		msg, err := factory.Create(Topic(parts[0]), Change(parts[1]), int(fmt), body)
		if err != nil {
			return nil, err
		}
		out = append(out, msg)
	}
	return out, nil
}
