package cartman

import (
	"context"
	"fmt"
	"net/http"
	"regexp"
)

type oauthToken string
type persistentToken string
type sudoToken string
type ownerID string
type clientID string
type extjwtToken string

var (
	oauthHeaderRE  = regexp.MustCompile("^OAuth\\s+(\\w+)$")
	extjwtHeaderRE = regexp.MustCompile("^Bearer\\s+([-_\\w]+[.][-_\\w]+[.][-_\\w]+)$")
)

const (
	// MockHeader is the name of the header which specifies an owner ID which
	// will be parsed and used for capabilities specifying an authn_user param
	MockHeader = "X-Mock-Owner-ID"
	// RailsHeader is the name of the header which specifies a Rails cookie
	RailsHeader = "Twitch-Persistent"
	// AuthorizationHeader is the name of the header which includes an OAuth token
	// or JWT tokens -- the latter will be prefaced with "Bearer "
	AuthorizationHeader = "Authorization"
	// ClientIDHeader is the name of the header which may contain the Client ID
	// necessary for extjwt tokens
	ClientIDHeader = "Client-ID"
	// ClientIDQueryParameter is the name of the query param which may contain
	// the Client ID necessary for extjwt tokens
	ClientIDQueryParameter = "client_id"
	// SudoHeader is the name of the header which includes a Sudo token
	SudoHeader = "Twitch-Sudo"
	// BitsSudoHeader is the name of the header which includes a Bits Sudo token
	BitsSudoHeader = "Twitch-Bits-Sudo"

	// CsrfCookie is the name of the cookie storing the value of the CSRF token
	CsrfCookie = "csrf_token"
	// CsrfHeader is the name of the header storing the value of the CSRF token
	CsrfHeader = "X-CSRF-Token"
	// OriginHeader is the name of the header storing the request Origin
	OriginHeader = "Origin"
	// MethodHeader is the name of the header storing the original HTTP method
	MethodHeader = "X-Origin-Http-Method"

	visageCtx = "AuthorizationOauth"
)

// addAuthentications extracts authentication tokens from the original request
// and adds them to the appropriate headers of the outgoing Cartman request.
func addAuthentications(ctx context.Context, h http.Header, origin *http.Request) {
	oauth := getOauth(ctx, origin)
	clientID, extjwt := getExtjwt(origin)
	persistent := getPersistent(origin)
	sudo := getSudo(origin)
	bitsSudo := getBitsSudo(origin)
	mockOwnerID := getMockOwnerID(origin)

	// note: don't allow both oauth and extjwt in the same request -- in addition
	// to the header collision, we don't want to potentially grant a combination
	// of privileges to a request that wouldn't be present with only one auth
	if oauth == nil && extjwt != nil && clientID != nil {
		h.Add(AuthorizationHeader, fmt.Sprintf("Bearer %s", string(*extjwt)))
		h.Add(ClientIDHeader, string(*clientID))
	}
	if oauth != nil {
		h.Add(AuthorizationHeader, fmt.Sprintf("OAuth %s", string(*oauth)))
	}
	if persistent != nil {
		h.Add(RailsHeader, string(*persistent))
	}
	if sudo != nil {
		h.Add(SudoHeader, string(*sudo))
	}
	if bitsSudo != nil {
		h.Add(BitsSudoHeader, string(*bitsSudo))
	}
	if mockOwnerID != nil {
		h.Add(MockHeader, string(*mockOwnerID))
	}
}

func addSecurityTokens(r *http.Request, origin *http.Request) {
	cHeader := origin.Header.Get(CsrfHeader)
	if cHeader != "" {
		r.Header.Add(CsrfHeader, cHeader)
	}
	oHeader := origin.Header.Get(OriginHeader)
	if oHeader != "" {
		r.Header.Add(OriginHeader, oHeader)
	}
	r.Header.Add(MethodHeader, origin.Method)

	c, e := origin.Cookie(CsrfCookie)
	if e == nil {
		r.AddCookie(c)
	}
}

func getMockOwnerID(req *http.Request) *ownerID {
	val := req.Header.Get(MockHeader)
	if val != "" {
		id := ownerID(val)
		return &id
	}
	return nil
}

func getExtjwt(req *http.Request) (*clientID, *extjwtToken) {
	cl := clientID(req.Header.Get(ClientIDHeader))
	if cl == "" {
		cl = clientID(req.URL.Query().Get(ClientIDQueryParameter))
		if cl == "" {
			return nil, nil
		}
	}

	auth := req.Header.Get(AuthorizationHeader)
	m := extjwtHeaderRE.FindStringSubmatch(auth)
	if m == nil {
		return nil, nil
	}

	et := extjwtToken(m[1])
	return &cl, &et
}

func getOauth(ctx context.Context, req *http.Request) *oauthToken {
	// There are four places the oauth token can come in
	// From https://github.com/justintv/Twitch-API/blob/master/authentication.md#authenticated-api-requests
	// - [Header] Authorization: OAuth <token>
	// - [Query String] oauth_token query parameter
	// - [Request Body] oauth_token as POST form value
	// From Visage:
	// - [Context] Oauth key
	var ot oauthToken

	// Check context
	if val, ok := ctx.Value(visageCtx).(string); ok {
		ot = oauthToken(val)
		return &ot
	}

	// Get oauth token from authorization header
	header := req.Header.Get(AuthorizationHeader)
	m := oauthHeaderRE.FindStringSubmatch(header)
	if m != nil {
		ot = oauthToken(m[1])
		return &ot
	}

	// Get oauth token from query string
	qs := req.URL.Query().Get("oauth_token")
	if qs != "" {
		ot = oauthToken(qs)
		return &ot
	}

	// See if it's in the POST form values
	err := req.ParseForm()
	if err != nil {
		return nil
	}

	tkn := req.Form.Get("oauth_token")
	if tkn != "" {
		ot = oauthToken(tkn)
		return &ot
	}

	return nil
}

func getPersistent(req *http.Request) *persistentToken {
	cookie, err := req.Cookie("persistent")
	if err != nil {
		return nil
	}
	var pt persistentToken

	pt = persistentToken(cookie.Value)
	return &pt
}

func getSudo(req *http.Request) *sudoToken {
	cookie, err := req.Cookie("sudo")
	if err != nil {
		return nil
	}
	var st sudoToken
	st = sudoToken(cookie.Value)
	return &st
}

func getBitsSudo(req *http.Request) *sudoToken {
	cookie, err := req.Cookie("bits_sudo")
	if err != nil {
		return nil
	}
	var st sudoToken
	st = sudoToken(cookie.Value)
	return &st
}
