package cartman

import (
	"context"
	"net/http"

	"code.justin.tv/foundation/twitchclient"
)

const (
	// BreakawayCapability verifies the login parameter against the owner of the
	// provided authentication credentials
	BreakawayCapability = "devsuccess::access_breakaway"

	// CartmanCapability is the default capability and only verifies
	// authentication credentials
	CartmanCapability = "cartman::authenticate_user"

	defaultStatSampleRate = 0.1
	defaultTimingXactName = "cartman"
)

// Client implements the API requests a client should be making
type Client interface {
	GetPermittedActions(ctx context.Context, rp RequestParams, reqOpts *twitchclient.ReqOpts) (*ActionsResponse, error)
	GetAuthorizationToken(ctx context.Context, rp RequestParams, reqOpts *twitchclient.ReqOpts) (*TokenResponse, error)
}

type client struct {
	twitchclient.Client
}

// NewClient initiates a twitchclient client
func NewClient(conf twitchclient.ClientConf) (Client, error) {
	if conf.TimingXactName == "" {
		conf.TimingXactName = defaultTimingXactName
	}
	twitchClient, err := twitchclient.NewClient(conf)
	return &client{twitchClient}, err
}

// HasAuthenticationCredentials is a convenience function that allows Cartman
// users to determine if a request is anonymous or not. Internally it reuses
// the addAuthentications code so that it will accurately reflect the actual
// credentials that will be forwarded without needing a second maintenance
// point.
func HasAuthenticationCredentials(ctx context.Context, origin *http.Request) bool {
	credentials := make(http.Header)
	addAuthentications(ctx, credentials, origin)
	return len(credentials) > 0
}
