package cartman

import (
	"context"
	"fmt"
	"net/http"
	"strings"
)

// RequestParams defines the arguments and parameters used to build a Cartman
// request
type RequestParams struct {
	// Scope is deprecated in favor of the Capabilities and KeyPath fields
	Scope string

	Capabilities []string
	OauthScopes  []string
	KeyPath      string
	Params       map[string]string
	Origin       *http.Request
}

const (
	// CapsParam is the name of the query parameter which specifies a list of
	// capabilities Cartman should verify on a user's behalf
	CapsParam = "capabilities"
	// CapsDelim is the string delimiter used to separate capabilities in the
	// query parameter
	CapsDelim = ","
	// KeyParam is the name of the query parameter which specifies either the
	// file name or file path of the signing key Cartman should use to sign an
	// authorization token
	KeyParam = "key"

	// ScopesParam is the name of the query parameter which specifies a list
	// of OAuth scopes to verify
	ScopesParam = "oauth_scopes"
	// ScopesDelim is the string delimiter used to separate OAuth scopes in the
	// query parameter
	ScopesDelim = ","
	// ScopeParam (DEPRECATED) is the name of the query parameter which specifies
	// a scope
	ScopeParam = "scope"
)

// BuildRequest constructs the HTTP request used to send a request to Cartman
// given the expected parameters
func (c *client) buildRequest(ctx context.Context, rp RequestParams, requestType string) (*http.Request, error) {
	if rp.Origin == nil {
		return nil, ErrMissingOrigin
	}

	var path string
	if rp.Scope != "" {
		path = fmt.Sprintf("/%s/%s", requestType, rp.Scope)
	} else {
		path = fmt.Sprintf("/%s", requestType)
	}

	req, err := c.NewRequest("GET", path, nil)
	if err != nil {
		return nil, err
	}

	req.URL.RawQuery = buildQueryString(req, rp)
	addAuthentications(ctx, req.Header, rp.Origin)
	addSecurityTokens(req, rp.Origin)
	return req, nil
}

// buildQueryString constructs the query string based on specified parameters
// in addition to the capabilities and signing key parameters
func buildQueryString(r *http.Request, p RequestParams) string {
	q := r.URL.Query()
	for k, v := range p.Params {
		q.Add(k, v)
	}

	if len(p.Capabilities) > 0 {
		q.Add(CapsParam, strings.Join(p.Capabilities, CapsDelim))
	}

	if len(p.OauthScopes) > 0 {
		q.Add(ScopesParam, strings.Join(p.OauthScopes, ScopesDelim))
	}

	if p.KeyPath != "" {
		q.Add(KeyParam, p.KeyPath)
	}
	return q.Encode()
}
