package main

import (
	"fmt"
	"log"
	"strconv"

	"github.com/aws/aws-sdk-go/aws/session"

	"sync"

	"code.justin.tv/chat/pushy/client/events"
	"code.justin.tv/common/config"
	"code.justin.tv/foundation/twitchclient"
	"code.justin.tv/foundation/twitchserver"
	"code.justin.tv/gds/gds/extensions/emailsender"
	ems "code.justin.tv/gds/gds/extensions/ems/client"
	"code.justin.tv/gds/gds/golibs/enterexit"
	"code.justin.tv/gds/gds/golibs/rollbar"
	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/service/sns"
	"github.com/aws/aws-sdk-go/service/sqs"
)

func init() {
	config.Register(map[string]string{
		"ems-url":                    "",
		"queue-url":                  "",
		"dispatch-arn":               "",
		"wait-time":                  "10", // in seconds
		"message-visibility-timeout": "30", // in seconds
		"num-of-workers":             "4",
	})
}

func main() {
	exitCode := 1
	defer enterexit.CleanExiter(&exitCode)()

	rollbar.InitInstance()

	err := config.Parse()
	if err != nil {
		rollbar.Error(rollbar.ERR, err)
		panic(err)
	}
	// start session
	queueURL := config.MustResolve("queue-url")
	waitTimeout, err := strconv.ParseInt(config.MustResolve("wait-time"), 10, 64)
	if err != nil {
		rollbar.Error(rollbar.ERR, err)
		log.Printf("Error parsing visibility timeout: %s", err)
	}

	visibilityTimeout, err := strconv.ParseInt(config.MustResolve("message-visibility-timeout"), 10, 64)
	if err != nil {
		rollbar.Error(rollbar.ERR, err)
		log.Printf("Error parsing visibility timeout: %s", err)
	}

	cfg := aws.NewConfig().WithRegion("us-west-2")
	sess, err := session.NewSession(cfg)
	if err != nil {
		rollbar.Error(rollbar.ERR, err)
		panic(err)
	}
	queue := emailsender.Queue{
		Client:            sqs.New(sess),
		URL:               queueURL,
		VisibilityTimeout: visibilityTimeout,
	}

	pushyClient, err := events.NewPublisher(events.Config{
		DispatchSNSARN: config.MustResolve("dispatch-arn"),
		Stats:          config.ClusterStatsd(),
		SNSClient:      sns.New(sess),
	})
	if err != nil {
		rollbar.Error(rollbar.ERR, err)
	}

	ems, err := ems.NewClient(twitchclient.ClientConf{
		Host: config.MustResolve("ems-url"),
	})
	if err != nil {
		rollbar.Error(rollbar.ERR, err)
		panic(err)
	}

	msgChan := make(chan *sqs.Message)
	numOfWorkers, _ := strconv.Atoi(config.MustResolve("num-of-workers"))
	for i := 0; i < numOfWorkers; i++ {
		worker := emailsender.NewWorker(pushyClient, ems, queue, config.ClusterStatsd(), msgChan)
		worker.Go()
	}

	twitchserver.AddDefaultSignalHandlers()
	go func() {
		err = twitchserver.ListenAndServe(twitchserver.NewServer(), nil)

		if err != nil {
			err = fmt.Errorf("ListenAndServe failed with: %s", err)
			rollbar.Error(rollbar.ERR, err)
			log.Fatal(err)
		}
	}()

	var wg sync.WaitGroup
	wg.Add(1)
	go func() {
		defer wg.Done()
		for {
			messages, err := queue.GetMessages(waitTimeout)
			if err != nil {
				log.Printf("Got error getting messages from queue: %s", err)
				continue
			}

			// if we got some, process messages
			numOfMessages := len(messages)
			if numOfMessages > 0 {
				for _, message := range messages {
					msgChan <- message
				}
			}
		}
	}()

	wg.Wait()
}
