package emailsender

import (
	"fmt"

	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/service/sqs"
	"github.com/aws/aws-sdk-go/service/sqs/sqsiface"
)

// Queue abstracts valuable attributes for handling sqs operations
type Queue struct {
	Client            sqsiface.SQSAPI
	URL               string
	VisibilityTimeout int64
}

// GetMessages retrieves messages off queue until the waitTimeout is elapsed, then returns those messages
func (q *Queue) GetMessages(waitTimeout int64) ([]*sqs.Message, error) {
	params := sqs.ReceiveMessageInput{
		QueueUrl:          aws.String(q.URL),
		VisibilityTimeout: aws.Int64(q.VisibilityTimeout),
	}
	if waitTimeout > 0 {
		params.WaitTimeSeconds = aws.Int64(waitTimeout)
	}
	resp, err := q.Client.ReceiveMessage(&params)
	if err != nil {
		return nil, fmt.Errorf("failed to get messages, %v", err)
	}

	return resp.Messages, nil
}

// DeleteMessage deletes the message corresponding to the provided receiptHandle from the queue
func (q *Queue) DeleteMessage(receiptHandle *string) error {
	_, err := q.Client.DeleteMessage(&sqs.DeleteMessageInput{
		QueueUrl:      aws.String(q.URL),
		ReceiptHandle: receiptHandle,
	})
	if err != nil {
		return err
	}
	return nil
}
