package emailsender

import (
	"context"
	"encoding/json"
	"fmt"
	"log"

	"code.justin.tv/chat/pushy/client/events"
	"code.justin.tv/common/twirp/hooks/statsd"
	ems "code.justin.tv/gds/gds/extensions/ems/client"
	"code.justin.tv/gds/gds/extensions/models"
	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/service/sqs"
)

const (
	developerCategory = "developer"
	onboardAction     = "onboard"
	offboardAction    = "offboard"
)

type messageBody struct {
	Type             string
	MessageId        string
	TopicArn         string
	Message          string
	Timestamp        string
	SignatureVersion string
	Signature        string
	SigningCertURL   string
	UnsubscribeURL   string
}

type messagePayload struct {
	ChannelID         string `json:"channel_id"`
	InviteID          string `json:"invite_id"`
	Timestamp         string `json:"timestamp"`
	Action            string `json:"action"`
	ActionCategory    string `json:"action_category"`
	Category          string `json:"category"`
	DeveloperCategory string `json:"developer_category"`
}

type Worker struct {
	pushyPublisher events.Publisher
	ems            ems.Client
	queue          Queue
	stats          statsd.Statter
	messageChan    chan *sqs.Message
	quit           chan bool
}

func NewWorker(pushyPublisher events.Publisher, ems ems.Client, queue Queue, stats statsd.Statter, messages chan *sqs.Message) *Worker {
	return &Worker{
		pushyPublisher: pushyPublisher,
		ems:            ems,
		queue:          queue,
		stats:          stats,
		messageChan:    messages,
		quit:           make(chan bool, 1),
	}
}

func (w *Worker) Go() {
	go func() {
		for {
			select {
			case message := <-w.messageChan:
				err := w.processMessage(message)
				if err != nil {
					log.Printf("got error processing messgage: %s", err)
				}
			case <-w.quit:
				return
			}
		}
	}()
}

// processMessage processes a message off the sns topic, whitelists a user, and sends them an email
// we delete the message off the queue, othe
func (w *Worker) processMessage(message *sqs.Message) error {
	var body messageBody
	err := json.Unmarshal([]byte(aws.StringValue(message.Body)), &body)
	if err != nil {
		w.stats.Inc("processmessage.unmarshal.body.err", 1, 0.1)
		log.Printf("Error unmarshalling body %v: %v", message.Body, err)
		// if we fail to parse, the message will get placed on a DLQ
		return err
	}
	var payload messagePayload
	err = json.Unmarshal([]byte(body.Message), &payload)
	if err != nil {
		w.stats.Inc("processmessage.unmarshal.payload.err", 1, 0.1)
		log.Printf("Error unmarshalling message %v: %v", message.Body, err)
		return err
	}

	// if message is not developer or is not a message corresponding to an onboard event, delete from queue and return
	if payload.ActionCategory != developerCategory {
		err = w.queue.DeleteMessage(message.ReceiptHandle)
		if err != nil {
			w.stats.Inc("processmessage.deletemessage.err", 1, 0.1)
			return err
		}

		log.Printf("WRONG ACTION CATEGORY: %v", payload)
		w.stats.Inc("processmessage.wrongcategory", 1, 0.1)
		return nil
	}

	if payload.Action == onboardAction {
		err = w.whitelistAndEmail(payload.ChannelID)
		if err != nil {
			return fmt.Errorf("Error whitelisting channelID %s: %s", payload.ChannelID, err)
		}
	}

	if payload.Action == offboardAction {
		err = w.unwhitelist(payload.ChannelID)
		if err != nil {
			return fmt.Errorf("Error unwhitelisting channelID %s: %s", payload.ChannelID, err)
		}
	}

	// delete message from queue
	log.Printf("successfully deleted message")
	err = w.queue.DeleteMessage(message.ReceiptHandle)
	if err != nil {
		w.stats.Inc("processmessage.deletemessage.err", 1, 0.1)
		return err
	}

	w.stats.Inc("processmessage.success", 1, 0.1)
	return nil
}

func (w *Worker) whitelistAndEmail(channelID string) error {
	isWhitelisted, err := w.ems.CanAddExtensions(context.Background(), channelID, nil)
	if err != nil {
		w.stats.Inc("processmessage.whitelist.datasource.err", 1, 0.1)
		return err
	}

	// if in whitelist log to already existing whitelist table
	if isWhitelisted {
		log.Printf("Already whitelisted developer")
		// todo figure out where to whitelist existings devs
		err = w.ems.WhitelistUser(context.Background(), channelID, models.ActionExisting, nil)
		if err != nil {
			w.stats.Inc("processmessage.existingwhitelist.err", 1, 0.1)
			return err
		}
		log.Printf("Added %s as existing whitelist user", channelID)
		w.stats.Inc("processmessage.existingwhitelist.err", 1, 0.1)
	}

	// if not, add to whitelist
	log.Printf("Whitelisted %s", channelID)
	err = w.ems.WhitelistUser(context.Background(), channelID, models.ActionAddExtension, nil)
	if err != nil {
		w.stats.Inc("processmessage.whitelist.err", 1, 0.1)
		return err
	}
	w.stats.Inc("processmessage.whitelist.success", 1, 0.1)

	// send email
	log.Printf("Sending email")
	err = w.pushyPublisher.Publish(context.Background(), events.DeveloperWhitelistedEvent, events.DeveloperWhitelistedParams{
		ChannelID: channelID,
	})
	if err != nil {
		w.stats.Inc("processmessage.whitelist.pushy.err", 1, 0.1)
		log.Printf("got error sending to pushy", err)
		return err
	}
	w.stats.Inc("processmessage.email.success", 1, 0.1)
	log.Printf("Email send to %s", channelID)
	return nil
}

func (w *Worker) unwhitelist(channelID string) error {
	// remove from whitelist
	log.Printf("unwhitelisted")
	err := w.ems.DeleteWhitelistUser(context.Background(), channelID, models.ActionAddExtension, nil)
	if err != nil {
		w.stats.Inc("processmessage.unwhitelist.err", 1, 0.1)
		return err
	}
	return nil
}
