package protocol

import (
	"strings"
)

type Address struct {
	ExtensionID string      `json:"extension_id"`
	SegmentType SegmentType `json:"segment"`
	ChannelID   string      `json:"channel_id,omitempty"`
}

// Segment converts an Address to a validated Segment
func (a *Address) Segment() (*segmentData, error) {
	switch a.SegmentType {
	case BroadcasterType:
		return Broadcaster(a.ChannelID)
	case DeveloperType:
		return Developer(a.ChannelID)
	case GlobalType:
		if a.ChannelID != "" {
			return nil, ErrIllegalSegmentChannel(a.SegmentType, a.ChannelID)
		}
		return Global(), nil
	}
	return nil, ErrUnknownSegmentType(a.SegmentType)
}

// AddressesForSegments returns the list of addresses necessary to
// load all of the input extensions for the indicated segments.
func AddressesForSegments(segments []Segment, extensionIDs ...string) []Address {
	out := make([]Address, 0, len(segments)*len(extensionIDs))
	for _, id := range extensionIDs {
		for _, segment := range segments {
			out = append(out, Address{id, segment.Type(), segment.ChannelID()})
		}
	}
	return out
}

// String converts an address to debug or serialized string format
func (a Address) String() string {
	fmt := a.ExtensionID + ":" + string(a.SegmentType)
	if a.ChannelID != "" {
		fmt = fmt + ":" + a.ChannelID
	}
	return fmt
}

// ParseAddress loads a string definition into an address; it is the inverse of
// the String function
func ParseAddress(fmt string) Address {
	segs := strings.Split(fmt, ":")
	var a Address
	a.ExtensionID = segs[0]
	if len(segs) > 1 {
		a.SegmentType = SegmentType(segs[1])
	} else {
		a.SegmentType = GlobalType
	}
	if len(segs) > 2 {
		a.ChannelID = segs[2]
	}
	return a
}
