package auth

import (
	barbradyTwirp "code.justin.tv/amzn/TwitchExtensionsBarbradyTwirp"
	"github.com/afex/hystrix-go/hystrix"
	statsdClient "github.com/cactus/go-statsd-client/statsd"
	"math/rand"
	"net/http"
	"time"

	ga "code.justin.tv/common/goauthorization"
	"code.justin.tv/extensions/fulton-configuration/protocol"
)

const (
	// CartmanAudience is the expected audience for capabilities signed by Cartman
	CartmanAudience = "code.justin.tv/extensions/fulton-configuration"
	// CartmanIssuer is the expected issuer for capabilities signed by Cartman
	CartmanIssuer = "code.justin.tv/web/cartman"
)

type jwtHandler struct {
	decoder                 ga.Decoder
	barbradyClient          barbradyTwirp.TwitchExtensionsBarbrady
	barbradySamplingPercent int64
	statsd                  statsdClient.Statter
}

// NewJWTHandler takes a decoder and whitelist and serves token based
// authorization. It uses the audience to determine if a token is coming from
// Cartman or GQL.
func NewJWTHandler(decoder ga.Decoder, barbradyClient barbradyTwirp.TwitchExtensionsBarbrady, barbradySamplingPercent int64, statsd statsdClient.Statter) Handler {
	hystrix.ConfigureCommand("barbrady_capability_lookup", hystrix.CommandConfig{
		ErrorPercentThreshold: 60,
		MaxConcurrentRequests: 4000,
		SleepWindow:           1000,
		Timeout:               250,
	})

	return &jwtHandler{decoder, barbradyClient, barbradySamplingPercent, statsd}
}

func (j *jwtHandler) GetCredentials(r *http.Request) (Credentials, error) {
	token, err := j.decoder.ParseToken(r)
	if err == ga.ErrNoAuthorizationToken {
		return NoPermissions(), nil // anonymous request
	}

	// we couldn't parse it, so reject it
	if err != nil {
		return nil, protocol.ErrUnauthorized
	}

	// check validity as a Cartman capability token
	err = j.decoder.Validate(token, ga.CapabilityClaims{})
	if err == nil {
		cartmanCreds := NewCartmanCredentials(j.decoder, token)

		if j.shouldUseBarbrady() {
			return NewBarbradyCredentials(cartmanCreds, j.barbradyClient, j.statsd), nil
		}

		return cartmanCreds, nil
	}

	// if it's not a valid token in any context, reject it
	return nil, protocol.ErrUnauthorized
}

func (j *jwtHandler) shouldUseBarbrady() bool {
	rand.Seed(time.Now().UnixNano())
	return int64(rand.Intn(100)) < j.barbradySamplingPercent
}
