package auth

import (
	statsdClient "github.com/cactus/go-statsd-client/statsd"
	"code.justin.tv/extensions/fulton-configuration/auth/fakes"
	"net/http"
	"testing"
	"time"

	ga "code.justin.tv/common/goauthorization"
	"code.justin.tv/foundation/twitchclient"
	"github.com/stretchr/testify/assert"
	"github.com/stretchr/testify/require"
)

func setToken(t *testing.T, params ga.TokenParams, r *http.Request) {
	encoder, err := ga.NewEncoder("HS512", CartmanIssuer, []byte("sekrat"))
	require.NoError(t, err)
	token := encoder.Encode(params)
	bytes, err := token.Serialize()
	require.NoError(t, err)
	r.Header.Set(twitchclient.TwitchAuthorizationHeader, string(bytes))
}

func TestNewJwtHandler(t *testing.T) {
	mockStatsd, err := statsdClient.NewClientWithConfig(&statsdClient.ClientConfig{
		Address:       ":8000",
		Prefix:        "mock_prefix",
		ResInterval:   0,
		UseBuffered:   false,
		FlushInterval: 0,
		FlushBytes:    0,
	})
	require.Nil(t, err)
	require.NotNil(t, mockStatsd)

	decoder, err := ga.NewDecoder("HS512", CartmanAudience, CartmanIssuer, []byte("sekrat"))
	require.NoError(t, err)
	handler := NewJWTHandler(decoder, &fakes.FakeTwitchExtensionsBarbrady{}, 100, mockStatsd)
	handlerNoBarbrady := NewJWTHandler(decoder, &fakes.FakeTwitchExtensionsBarbrady{}, 0, mockStatsd)

	t.Run(".GetCredentials", func(t *testing.T) {
		t.Run("should treat requests without tokens as anonymous", func(t *testing.T) {
			r, err := http.NewRequest("GET", "/", nil)
			require.NoError(t, err)
			creds, err := handler.GetCredentials(r)
			assert.NoError(t, err)
			assert.Equal(t, NoPermissions(), creds)
			creds, err = handlerNoBarbrady.GetCredentials(r)
			assert.NoError(t, err)
			assert.Equal(t, NoPermissions(), creds)
		})

		t.Run("should treat tokens with expected Cartman audience as Cartman calls", func(t *testing.T) {
			r, err := http.NewRequest("GET", "/", nil)
			require.NoError(t, err)
			var params ga.TokenParams
			params.Exp = time.Now().Add(time.Minute)
			params.Aud = []string{CartmanAudience}
			setToken(t, params, r)

			creds, err := handler.GetCredentials(r)
			assert.NoError(t, err)
			assert.IsType(t, &BarbradyCredentials{}, creds)

			creds, err = handlerNoBarbrady.GetCredentials(r)
			assert.NoError(t, err)
			assert.IsType(t, &cartmanCredentials{}, creds)
		})

		t.Run("should return an error if a token is expired", func(t *testing.T) {
			r, err := http.NewRequest("GET", "/", nil)
			require.NoError(t, err)
			var params ga.TokenParams
			params.Exp = time.Now().Add(-time.Minute)
			params.Aud = []string{CartmanAudience}
			setToken(t, params, r)

			creds, err := handler.GetCredentials(r)
			assert.Nil(t, creds)
			assert.Error(t, err)

			creds, err = handlerNoBarbrady.GetCredentials(r)
			assert.Nil(t, creds)
			assert.Error(t, err)
		})

		t.Run("should return an error if a token is not correctly structured", func(t *testing.T) {
			r, err := http.NewRequest("GET", "/", nil)
			require.NoError(t, err)
			r.Header.Set(twitchclient.TwitchAuthorizationHeader, "very.broken.token")

			creds, err := handler.GetCredentials(r)
			assert.Nil(t, creds)
			assert.Error(t, err)

			creds, err = handlerNoBarbrady.GetCredentials(r)
			assert.Nil(t, creds)
			assert.Error(t, err)
		})
	})
}
