package auth

import (
	"context"
	"net/http"

	"github.com/jixwanwang/apiutils"
)

var (
	// actual key is the address of this var, which is unique in program space
	credentialsValue = -42
	credentialsKey   = &credentialsValue
)

// StoreHandler builds a goji middleware that automatically places a
// Credentials into the context of each request or early outs with an
// unauthorized error.
func StoreHandler(handler Handler) func(http.Handler) http.Handler {
	return func(inner http.Handler) http.Handler {
		return http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
			creds, err := handler.GetCredentials(r)
			if err != nil {
				apiutils.ServeError(w, apiutils.NewErrorResponse(http.StatusUnauthorized, err.Error()))
			} else {
				ctx := Store(r.Context(), creds)
				inner.ServeHTTP(w, r.WithContext(ctx))
			}
		})
	}
}

// Store wraps the input context with one containing the specified credentials
func Store(ctx context.Context, creds Credentials) context.Context {
	return context.WithValue(ctx, credentialsKey, creds)
}

// Load retrieves credentials from a request context.
func Load(ctx context.Context) Credentials {
	if value, ok := ctx.Value(credentialsKey).(Credentials); ok {
		return value
	}
	return NoPermissions()
}
