package auth

import (
	"context"
	"errors"
	"net/http"
	"testing"

	"code.justin.tv/gds/gds/golibs/mocks"
	"github.com/stretchr/testify/assert"
	"github.com/stretchr/testify/require"
	goji "goji.io"
	"goji.io/pat"
)

var errExpected = errors.New("expected")

type badHandler struct{}

func (badHandler) GetCredentials(_ *http.Request) (Credentials, error) {
	return nil, errExpected
}

func TestLoad(t *testing.T) {
	t.Run("should return NoPermissions() by default", func(t *testing.T) {
		assert.Equal(t, NoPermissions(), Load(context.Background()))
	})

	t.Run("should return stored credentials if present", func(t *testing.T) {
		creds := AllPermissions()
		ctx := context.WithValue(context.Background(), credentialsKey, creds)
		assert.Equal(t, creds, Load(ctx))
	})
}

func TestStore(t *testing.T) {
	t.Run("should place credentials into a context", func(t *testing.T) {
		creds := AllPermissions()
		ctx := Store(context.Background(), creds)
		assert.Equal(t, creds, ctx.Value(credentialsKey))
	})
}

func TestStoreHandler_Success(t *testing.T) {
	handler := &FakeHandler{}
	inner := http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
		creds, err := handler.GetCredentials(r)
		assert.NoError(t, err)
		assert.Equal(t, creds, Load(r.Context()))
		w.WriteHeader(200)
	})

	server := goji.NewMux()
	server.Use(StoreHandler(handler))
	server.Handle(pat.Get("/"), inner)

	writer := mocks.NewResponseWriter()
	req, err := http.NewRequest("GET", "/", nil)
	require.NoError(t, err)
	server.ServeHTTP(writer, req)
	assert.Equal(t, writer.LastStatus(), 200)

	writer = mocks.NewResponseWriter()
	handler.Credentials = AllPermissions()
	server.ServeHTTP(writer, req)
	assert.Equal(t, writer.LastStatus(), 200)
}

func TestStoreHandler_Failure(t *testing.T) {
	inner := http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
		assert.Fail(t, "callback should not be used without credentials")
		w.WriteHeader(200)
	})

	server := goji.NewMux()
	server.Use(StoreHandler(badHandler{}))
	server.Handle(pat.Get("/"), inner)

	writer := mocks.NewResponseWriter()
	req, err := http.NewRequest("GET", "/", nil)
	require.NoError(t, err)
	server.ServeHTTP(writer, req)
	assert.Equal(t, writer.LastStatus(), 401)
}
